package vcb.engine.tron;

import java.util.HashMap;

/**
* This defines "conversions" the parser can use for custom
* conversions of xml into a datatron. The purpose is to give
* datatrons that are easier for business logic to use.
* <p>
* The only conversion currently supported converts a list of
* duplicate container tags to unique tags, using a child tag and
* value per container tag. For example suppose we need to convert: 
* <pre>
* <Pin>
*     <PinName>GetCustomerAddress</PinName>
*     <Intent>Get</Intent>
*     <Fields>
*         <Field>
*             <Name>CustomerAddress</Name>
*             <Type>String</Type>
*             <IsProvided>true</IsProvided>
*             </Field>
*         <Field>
*             <Name>CustomerAge</Name>
*             <Type>int</Type>
*             </Field>
*         </Fields>
*     </Pin>
* <Pin>
*     <PinName>GetText</PinName>
*     <Intent>Get</Intent>
*     </Pin>
* <pre>
* to the following structure, where the Pin and Field list tags
* become unique tags, using PinName and Name:
* <pre>
* <GetCustomerAddress>
*     <Intent>Get</Intent>
*     <Fields>
*         <CustomerAddress>
*             <Type>String</Type>
*             <IsProvided>true</IsProvided>
*             </CustomerAddress>
*         <CustomerAge>
*             <Type>int</Type>
*             </CustomerAge>
*         </Fields>
*     </GetCustomerAddress>
* <GetText>
*     <Intent>Get</Intent>
*     </GetText>
* <pre>
* This conversion is accomplished by adding two conversions to
* this class, like this:
* <pre>
*   parserConversions.addListChild("Pin", "PinName");
*   parserConversions.addListChild("Field", "Name");
* </pre>
* This class is then passed to the parser with the xml String to
* parse, shown in the first example above. The parser will return
* a datatron representing the second above xml example.
*
* NOTE - No longer used in VCB, as of February 2001. <-----<<<
*
* @author Jack Harich
*/
public class ParserConversions {

//---------- Internal Fields -------------------------------------
protected HashMap conversions = new HashMap();

//---------- Public Methods --------------------------------------
/**
* Adds the "list child" conversion. The listTag must be unique
* per instance of this class. List tags having the child tag
* will be replaced with the child tag value and the child tag
* and value will be removed.
*/
public void addListChild(String listTag, String childTag) {
    if (conversions.containsKey(listTag)) throw new
        IllegalStateException("The listTag " + listTag + 
        " has already been added.");
    conversions.put(listTag, childTag); 
}
/**
* Returns the child tag for the listTag or null if none.
*/
public String getListChild(String listTag) {
    return (String)conversions.get(listTag);
}
//---------- Standard --------------------------------------------
private static void print(String text) {
    System.out.println("ParserConversions" + text);
}

} // End class