package vcb.engine.tron;

import java.util.Iterator;
import java.util.NoSuchElementException;

/**
* This is a collection of Entry objects, accessable by key or
* index. If functions as a map with ordered keys or as a list of
* ordered entries. Each key has an Entry as its value.
* <p>
* <b>Note that once the isList property goes true, gets and sets
* will throw an exception.</b> 
* <p>
* This class is for datatron partition internal use only.
*
* @author Jack Harich
*/
public class EntryMapList {

//---------- Internal Fields -------------------------------------
// Later we may use a Map and double linked list of entries
// This was my first naive implementation :-)
protected static final int INITAL_ENTRIES = 5;
protected Entry[] entries = new Entry[INITAL_ENTRIES];
protected int     size = 0; // Number of entries in array

protected boolean isList; // true if dupe keys present

//---------- Superclass Overrides --------------------------------
/**
* Determines if this equals the object. Returns false if the object
* is null or not an EntryMapList. Returns false if their isList
* properties are not equal. If isList then does full comparison
* with same order required. If not isList then does a map
* comparison where order is irrelevant. The last is useful for
* comparing two maps and not worrying about their order.
*/
public boolean equals(Object object) {
    if (object == null) {
        return false;
    } else if (! (object instanceof EntryMapList) ) {
        return false;
    } else {
        EntryMapList entryMapList = (EntryMapList)object;
        // Compare isList
        if (! entryMapList.isList == isList) return false;
        // Compare entries 
        if (isList) {
            // ArrayList.equals() does same order check
            return entryMapList.entries.equals(entries);
        } else {
            // Check map equivalent
            return isEquivalentMap(entryMapList);
        }
    }
}
/**
* Returns a hash code int that is consistent if the keys, their
* sequence and the primitive values and String values are 
* consistent. To ensure this always call
* <code>sortKeys()</code> before using this class as a hash key.
*/
public int hashCode() {
    StringBuffer text = new StringBuffer();
    for (int i = 0; i < size; i++) {
        text.append(entries[i].key + "=" + entries[i].getHashCodeStringForValue() + ",");
    }
    return text.toString().hashCode();
}
//---------- Public Methods --------------------------------------
/**
* Sorts the keys into alphabetical order. This must be done before
* an instance of this class can be reliably used as a storage or
* getter key in a map, because the method hashCode() returns a
* value dependent on the concatenation of all keys and values. We
* see no other way to use this class reliably as a key.
*/
public void sortKeys() {
    if (size <= 1) return; // Nothing to sort
    
    // Inefficient but simple bubble sort. This method is seldom
    // called and then number of keys is usually small, so
    // efficiency is not a concern.
    // Could use Quicksort in org.jcon.util.ArraySorter.
    Entry swapEntry; // For swap
    boolean inOrder = false;
    
    while (! inOrder) {
        inOrder = true; // Set false if swapped
        for (int i = 0; i < size - 1; i++) {
            if (entries[i].key.compareTo(entries[i + 1].key) > 0) {
                // Out of order so swap
                swapEntry = entries[i];
                entries[i] = entries[i + 1];
                entries[i + 1] = swapEntry;
                inOrder = false;
            }        
        }
    }
}
/**
* Determines whether each key in this class has a value that's
* the same as that key's value in the supplied EntryMapList.
* This is useful in cases where
* entry order can be different but map content can be the same.
* This method is not useful for duplicate keys, because
* that is a list, not a map.
*/
public boolean isEquivalentMap(EntryMapList entryMapList) {
    if (entryMapList == null) return false;
    if (entryMapList.size() != size()) return false;
    // Compare each entry
    for (int i = 0; i < size; i++) {
        Entry otherEntry = entryMapList.getEntry(entries[i].key);
        if (otherEntry == null) return false;
        if (! otherEntry.equals(entries[i])) return false;
    }
    return true;
}
//----- The key tricky accessors
/**
* Adds the entry to the end of the collection. This also sets the
* "isList" flag to true if there are now duplicate keys.
*/
public void addEntry(Entry entry) {
    // Maintain isList
    if (size > 0 && ! isList) {
        if (firstIndexOf(entry.key) >= 0) isList = true;
    }
    // Add entry to end of entries
    addEntryAtEnd(entry);    
}
/**
* Puts the entry in the collection at the end since the caller
* knows the key is not a duplicate. <code>setEntry(Entry)</code>
* does the same thing but slower since it has to first check 
* for a duplicate key. For package use only. Used by Datatron.
*/
void setNonDuplicateKeyEntry(Entry entry) {
    if (isList) throw new IllegalStateException
        ("Cannot set entry because isList = true. The entry = " + entry);
        
    addEntryAtEnd(entry);
}
protected void addEntryAtEnd(Entry entry) {    
    expandIfFull();
    entries[size] = entry;
    size++;
}
/**
* Puts the entry in the collection at the first matching key or
* at the end if no matching key found. 
* Throws an IllegalStateException if isList is true.
*/
public void setEntry(Entry entry) {
    if (isList) throw new IllegalStateException
        ("Cannot set entry because isList = true. The entry = " + entry);

    int index = firstIndexOf(entry.key);
    if (index >= 0 ) {    
        // Found so replace value
        entries[index] = entry;
    } else {
        // Not found so add to end
        addEntryAtEnd(entry);
    }
}
/**
* Returns the first entry with the key or null if none.
* Throws an IllegalStateException if isList is true.
*/
public Entry getEntry(String key) {
    if (isList) throw new IllegalStateException
        ("Cannot get entry with key = '" + key + "' because isList = true.");

    int index = firstIndexOf(key);
    return (index >= 0 ? entries[index] : null);
}
/**
* Returns the first index of the key or -1 if not found.
*/
protected int firstIndexOf(String key) {
    for (int i = 0; i < size; i++) {
        if (entries[i].key.equals(key)) {
            return i;
        }
    }
    return -1; // Not found
}
//----- General
/**
* Determines whether this is possibly a list or not, in other
* words whether duplicate keys may be present. This property
* goes true when the first duplicate key is added.
* It goes false when the collection is emptied or the last
* duplicate key is removed.
* <p>
* <b>Note that once the isList property goes true, gets and sets
* will throw an exception.</b> 
*/
public boolean isList() {
    return isList;
}
/**
* Returns the size of the collection, useful for iteration.
*/
public int size() {
    return size;
}
/**
* Returns the entry reader at the <b>base one</b> index. 
* If the index is out of bounds throws an IndexOutOfBoundsException.
* Note this can be used for isList equals true or false. 
*/
public EntryReader getEntryAt(int index) {
    // Convert index to base zero - Until we convert to base one throughout
    index--;
    // Check bounds
    if (index < 0 || index > size - 1) {
        throw new IndexOutOfBoundsException("Index '" + (index + 1)
         + "' is out of bounds. My size is '" + size + "'.");
    }
    // Okay
    return (EntryReader)entries[index];
}
/**
* Returns an interation of EntryReader instances for read only
* access to each entry in the collection. Remove is not allowed.
* @see vcb.engine.tron.EntryReader
* @see vcb.engine.tron.DatatronIterator
*/
public DatatronIterator iterateEntries() {
    return new DatatronIterator(entries, size);
}
/**
* Determines if the collection is empty.
*/ 
public boolean isEmpty() {
    return size == 0;
}
/**
* Determines if the collection contains the key.
*/
public boolean containsKey(String key) {
    return (getEntry(key) != null ? true : false);
}
/**
* Removes and returns the entry at the index. Also collapses the
* underlying list so that no gaps appear. Also maintains isList.
*/
protected Entry removeKeyAt(int index) {
    if (index < 0 || index > size - 1) {
        throw new IndexOutOfBoundsException("The index is out of bounds. Current size is " +
            size + ".");
    }
    // Save entry we're about to remove
    Entry entry = entries[index];
    
    // Remove entry by copying over it, decrement size
    try {
        System.arraycopy(entries, index + 1, entries, index, size - index);
    } catch(Exception ex) {
        print(".removeKeyAt() - Failed to do arraycopy(), index = " + index + 
            ", entry = " + entry);
        ex.printStackTrace();
        throw new RuntimeException("Failure");
    }
    entries[size] = null;
    size--;
    resetIsList();
    
    // Done
    return entry;
}
protected void resetIsList() {
    if (size <= 1) {
        isList = false;
        return;
    }
    // Check each key for duplicate key not equal to itself
    for (int i = 0; i < size; i++) {
        String key = entries[i].key;
        for (int j = i + 1; j < size; j++) {
            if (key.equals(entries[j].key)) {
                isList = true;
                return;
            }
        }
    }
    isList = false; // No dupes
}
/**
* Adds all entries in the supplied EntryMapList to this one.
* Note these are added, not set. The added entries are cloned.
* The object, if any, in the entries in not cloned.
* Returns this EntryMapList.
*/
public EntryMapList addAll(EntryMapList mapList) {
    for (int i = 0; i < mapList.size; i++) {
        addEntry((Entry)mapList.entries[i].clone());
    }
    return this;
}
/**
* Returns the first Entry or null if none. One use for this
* is getting the entry in a datatron with only one entry.
*/
public Entry getFirstEntry() {
    return (size == 0 ? null : entries[0]);
}
//----- Removers
/**
* Removes all entries in the collection.
*/
public void clear() {
    for (int i = 0; i < size; i++) {
        entries[i].release();
        entries[i] = null;
    }
    // If array is enlarged slim it down
    if (entries.length > INITAL_ENTRIES) {
        entries = new Entry[INITAL_ENTRIES];
    }
    size = 0;
    isList = false;
}
/**
* Removes and returns the first entry with the key. If the key is
* not found then nothing is removed and null is returned.
*/
public Entry removeKey(String key) {
    for (int i = 0; i < size; i++) {
        if (entries[i].key.equals(key)) {
            return removeKeyAt(i);
        }
    }
    return null; // Not found
}
//---------- Package Methods -------------------------------------
/**
* Returns the Entry at the index, useful for interation.
*/
//Entry getEntryAt(int index) {
//    if (index < 0 || index > size - 1) {
//        complain("The index is out of bounds. Current size is " +
//            size + ".");
//        return null; // Never reached
//    } else {
//        return entries[index];
//    }
//}
//---------- Protected Methods -----------------------------------
// Note size remains unchanged
protected void expandIfFull() {
    if (size >= entries.length) { // Could use "==" but being cautious
        int oldLength = entries.length;
        Entry[] newArray = new Entry[oldLength * 2];
        System.arraycopy(entries, 0, newArray, 0, oldLength);
        entries = newArray;
    }
}
protected void complain(String text) {
    // We shall see if toString() is too verbose, maybe just name
    throw new IllegalStateException(text + " EntryMapList=" + 
        toString() );
}
//---------- Standard --------------------------------------------
private static void print(String text) {
    System.out.println("EntryMapList" + text);
}
//========== Inner Classes =======================================
public class MyIterator implements Iterator {
    protected int index; // Start at zero
    
    public boolean hasNext() {
        return (index < size() ? true : false);
    }
    public Object next() {
        if (index < size()) {
            return entries[index++];
        } else {
            throw new NoSuchElementException("No more entries.");
        }
    }
    public void remove() { // Perhaps later allow this
        throw new UnsupportedOperationException("Remove not allowed.");
    }
}

} // End class