package jsl.fw1.engine;

import vcb.inv.core.ClassResourceReader;
import vcb.engine.tron.Datatron;
import vcb.engine.tron.Parser;

/**
* This handles all DK storage. Currently this is just reads using
* a marker class and a resource path.
*
* @author Jack Harich
*/
public class DKStore {

//---------- Internal Fields -------------------------------------
protected ClassResourceReader reader = new ClassResourceReader();
protected Parser              parser = new Parser();

protected static final String CONTAINER_DK_FILE_NAME = "_Container.dk";
protected static final String LEAF_DK_FILE_TYPE = "dk";

//---------- Public Methods --------------------------------------
/**
* Provides the node with its DK. This method asserts the node needs
* DK. If it needs DK then the node must implement
* <code>setDK()</code>.
* <p>
* Containers may be reused. If the node is a Container and has a
* DK URL, then the DK URL is used to supply the container's DK
* instead of the default approach. The URL must be of the format
* "clrs://jsl.fw1.containers.ContainersMarker/test_container/xx".
*/
public void provideNodeDK(Node node) {
    // Assert node needs DK
    if (! node.isDKDriven()) return;
    
    // Check for container DK URL option
    if (node instanceof Container && node.getDKURL() != null) {
        provideContainerDKURL(node);
    } else {
        provideDKNormally(node);
    }
}
//---------- Protected Methods -----------------------------------
protected void provideContainerDKURL(Node node) { 
    String url = node.getDKURL();
    
    // Assert we can handle the protocol
    if (! url.startsWith("clrs")) throw new IllegalArgumentException
        ("Only the 'clrs' protocol is supported. URL " + url + " is not allowed.");
        
    // Parse URL into marker and resource path
    // First get marker
    int index = url.indexOf("//");
    url = url.substring(index + 2);
    index = url.indexOf("/");
    String marker = url.substring(0, index);
    // Next get resource
    String resource = url.substring(index + 1) + "/" + CONTAINER_DK_FILE_NAME;
    
    // Get DK using marker and resource
    print(".provideContainerDKURL() - marker = " + marker + 
        ", resource = " + resource);
    node.setDK(getDK(marker, resource));
} 
protected void provideDKNormally(Node node) {   
    // Calculate resource path to first marker found
    String resource = CONTAINER_DK_FILE_NAME;
    Node parent = node;
    if (node instanceof Leaf) {
        resource = node.getName() + "." + LEAF_DK_FILE_TYPE;
        parent = node.getParent();
    }
    String marker = parent.getMarker();
    
    while(marker == null) {
        resource = parent.getName() + "/" + resource;
        parent = parent.getParent();
        marker = parent.getMarker();
    }
    // The rest is easy :-)
    print(".provideDKNormally() - marker = " + marker + 
        ", resource = " + resource);
    node.setDK(getDK(marker, resource));
}
protected Datatron getDK(String markerClass, String resource) {
    String text = reader.readText(markerClass, resource);
    return parser.parseXMLtoDatatron(text);
}
//---------- Standard --------------------------------------------
private static void print(String text) {
    System.out.println("DKStore" + text);
}

} // End class