package vcb.inv.core;

import java.io.BufferedReader;
import java.io.File;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.IOException;

/**
* This class handles reads for class resource type persistence.
* This involves reading files using a class marker
* and resource name, aka file name. The file name is relative to
* the class marker location on disk.
* <p> 
* By using a class marker, not a class name, we preserve class
* loader location, allowing things such as applets to use
* class resource text off the web.
* <p>
* We use a separate class for reader to allow a smaller
* deployment package where writes are not needed, and to simplify
* the reader and writer classes.
*
* @author Jack Harich
*/
public class ClassResourceReader {

//---------- Public Methods --------------------------------------
/**
* A convenience method for <code>readText(Class, String)</code>.
* It turns the classResourceURL into a class and resource, and
* calls the other method.
*
* @param classResourceURL  the url to use. It must be of the
*   format <code>classname/path/filename</code> such as
*   <code>com.sys1.admin.System/admin/ap/Payables.dk</code>.
* @exception  IllegalArgumentException if the marker class 
*                cannot be created.
* @exception  IllegalArgumentException if marker or resource are null.
* @exception  IllegalStateException if file not found or unable to read.
*/
public String readText(String classResourceURL) {
    //***int firstSlashIndex = classResourceURL.indexOf("/");
    int firstSlashIndex = classResourceURL.indexOf(File.separator);
    // File.separator
    String className;
    try {
        className = classResourceURL.substring(0, firstSlashIndex);
    } catch(Exception ex) {
        throw new IllegalArgumentException("Unable to extract className from '" + 
            classResourceURL + "'. File.separator = '" + File.separator + "'.");
    }
    String resource  = classResourceURL.substring(firstSlashIndex + 1);
    //print(".readText() - className = " + className + 
    //    ", resource = " + resource);
        
    // Remove extra lines after testing
    String text = readText(className, resource);
    //print(" - text:\n" + text);
    return text;
}
/**
* A convenience method for <code>readText(Class, String)</code>.
* It turns the markerClassName into a class and call the other
* method.
*
* @param markerClassName  the marker class name to use.
* @param resource  the file name, aka resource, to read relative to
*                  the marker. It may have directories prepended to
*                  the file name. An example of a resource is:
*                  "admin/ap/Payables.dk".
* @exception  IllegalArgumentException if the marker class 
*                cannot be created.
* @exception  IllegalArgumentException if marker or resource are null.
* @exception  IllegalStateException if file not found or unable to read.
*/
public String readText(String markerClassName, String resource) {
    Class marker;
    try {
        marker = Class.forName(markerClassName);
    } catch(Exception ex) {
        throw new IllegalArgumentException("Cannot create class "
            + markerClassName + ".\n" + ex.getMessage() );
    }
    return readText(marker, resource);
}
/**
* Reads the text in the resource for the marker. This is done with
* the useful <code>Class.getResourseAsStream()</code>.
*
* @param marker  the class marker to use.
* @param resource  the file name, aka resource, to read relative to
*                  the marker. It may have directories prepended to
*                  the file name. An example of a resource is:
*                  "/ap/Payables.dk".
* @exception  IllegalArgumentException if marker or resource are null.
* @exception  IllegalStateException if file not found or unable to read.
*/
public String readText(Class marker, String resource) {

    if (marker == null) throw new IllegalArgumentException    
        ("The class marker cannot be null.");
    if (resource == null) throw new IllegalArgumentException    
        ("The resource cannot be null.");
        
    BufferedReader reader = openForInput(marker, resource);
        
    // Suprisingly, initial capacity of 30000 doesn't speed up
    // 10240 slightly better for large file
    StringBuffer text = new StringBuffer();
    try {
        String line = reader.readLine();
        while (line != null) {
            text.append(line);
            text.append("\n");
            line = reader.readLine();
        }
        reader.close();
    } catch(IOException ex) {
        throw new IllegalStateException("Cannot read text in: " +
        getDescription(marker, resource));
    }
    return text.toString();   
}
//---------- Protected Methods -----------------------------------
protected BufferedReader openForInput(Class marker, String resource) {
    
    InputStream inputStream 
        = marker.getResourceAsStream(resource);

    if (inputStream == null) throw new IllegalStateException
        ("Cannot open resource in: " + getDescription(marker, resource));
    
    return new BufferedReader(new InputStreamReader(inputStream));
} 
protected String getDescription(Class marker, String resource) {
    return marker.getName() + "/" + resource;
}
//---------- Standard --------------------------------------------
private static void print(String text) {
    System.out.println("ClassResourceReader" + text);
}

} // End class