package vcb.engine.tron;

/**
* This contains various static methods related to Datatrons.
* Note these are called only by classes in this package. 
*
* @author Jack Harich
*/
public class DatatronLib {

//---------- Public Methods --------------------------------------
/**
* Returns a String representation of the datatron's entries in a 
* tree format, since often the contents are a tree.
*/
public static String toString(EntryMapList entries) {
    StringBuffer text = new StringBuffer();
    buildTree(0, text, "Irrelevant", entries); // Root name not used
    if (text.length() == 0) return "";
    // Remove last \n
    String text2 = text.toString();    
    return text2.substring(0, text2.length() - 1);
}
/**
* Returns the datatron in XML format, using the entries and rootTag.
* If the rootTag is null then the xml is not wrapped in a root tag.
*/
public static String toXML(EntryMapList entries, String rootTag) {
    StringBuffer text = new StringBuffer();
    buildXMLTree(0, text, rootTag, entries); // Root name not used
    return text.toString();    
}
//---------- Protected Methods -----------------------------------
// toXML() helper, RECURSIVE
protected static void buildXMLTree(int level, StringBuffer text, 
                String dtName, EntryMapList entries) {
    if (level == 0) {
        if (dtName != null) text.append("<" + dtName + ">" + "\n");
        buildXMLTree(level + 1, text, dtName, entries); // RECURSE
        buildTreePrefix(2, text);
        if (dtName != null) {
            text.append("</" + dtName + ">"); // No "\n" since last line
            // Could remove last \n - Future feature ***
        }
    } else {
        // Handle children
        level ++; // Indent children
        DatatronIterator iterator = entries.iterateEntries();
        while (iterator.hasNext()) {
            EntryReader entry = iterator.nextEntry();
            String key = entry.getKey();
            // Consider Datatron nesting
            if (entry.getType() == Datatron.class) {
                // Container - RECURSE
                buildTreePrefix(level, text);
                text.append("<" + key + ">" + "\n");
                    buildXMLTree(level, text, key, entry.getDatatron().getEntries()); // RECURSE
                buildTreePrefix(level + 1, text);
                text.append("</" + key + ">" + "\n");
            } else {
                // Leaf 
                buildTreePrefix(level, text);
                text.append("<" + key + ">");
                    if (! entry.isNull()) text.append(entry.valueToString());
                text.append("</" + key + ">" + "\n");
            }
        }
    }              
}                
// toString() helper, RECURSIVE
protected static void buildTree(int level, StringBuffer text, 
                String dtName, EntryMapList entries) {
    if (level > 0) { // Don't include root name
        buildTreePrefix(level, text);
        text.append("+" + dtName + ":");
        text.append("\n");
    }
    level ++; // Indent children
    DatatronIterator iterator = entries.iterateEntries();
    while (iterator.hasNext()) {
        EntryReader entry = iterator.nextEntry();
        String key = entry.getKey();
        // Consider Datatron nesting
        if (entry.getType() == Datatron.class) {
            // Container - RECURSE
            buildTree(level, text, key, entry.getDatatron().getEntries());
        } else {
            // Leaf - append prefix, key, value
            buildTreePrefix(level, text);
            text.append(entry.toString() + "\n");
        }
    }
}
// toString() and toXML() support
protected static void buildTreePrefix(int level, StringBuffer text) {
    // Indent using level - Add prefix of spaces, none for level 1
    for (int j = 1; j <= level - 1; j++) {
        text.append("    "); // 4 spaces per level
    }
}
//---------- Standard --------------------------------------------
private static void print(String text) {
    System.out.println("DatatronLib" + text);
}

} // End class