package vcb.gen.util;

import java.util.Iterator;
import java.util.ConcurrentModificationException;

/**
* This is a one based list. One based means that all calls 
* involving "index" must range from one to the size of the list.
<p>
* When it come to "things", machines are zero based but people are
* one based. The use of zero and one based in the same line of
* code (such as for loops) is gone forever when using this class, 
* as are other related bugaboos. "Off by one" bugs are reduced. 
* To reduce them further, use iteration instead of a for loop.
* <p>
* This will eventually implement Collection and perhaps List. 
* Null elements are not supported. Duplicate elements are supported.
* <p>
* Method documentation to be added once vision is complete :-)
*
* @author Jack Harich
*/   // import vcb.gen.util.ListOne;
public class ListOne implements Cloneable {

//---------- Internal Fields -------------------------------------
protected Object[] list = new Object[10]; 
protected int      size = 0; // Equals next available index
protected int      modCount = 0; // Incremented on all mutations

//---------- Object Overrides ------------------------------------
public boolean equals(Object object) {
	if (object == this) return true;
	if (!(object instanceof ListOne)) return false;

    ListOne otherList = (ListOne)object;
    if (size != otherList.size) return false;
    
    for (int i = size; i >=1; i--) {
        if (! list[i - 1].equals(otherList.list[i - 1])) return false;
    }
    return true;
}
public int hashCode() {
	int hashCode = 1;
    for (int i = size; i >=1; i--) {
        // Per the List.hashCode() spec, without iterator or null
	    hashCode += 31*hashCode + list[i - 1].hashCode();
	}
	return hashCode;
}
public Object clone() {
	try { 
	    ListOne newList = (ListOne)super.clone();
	    newList.list = (Object[])list.clone();
	    newList.modCount = 0;
	    return newList;
	} catch (CloneNotSupportedException ex) { 
	    // Can't happen, since we are Cloneable and so are arrays
	    throw new InternalError(ex.toString());
	}
}
//---------- Collection Implementation ---------------------------
public boolean add(Object element) {
    if (element == null) throw new IllegalArgumentException("Null elements not supported.");
    
    if (size + 1 > list.length) increaseCapacity();
    
    list[size] = element;
    size++;
    modCount++;
    return true;
}
public boolean addAll(ListOne otherList) {
    Iterator iterator = otherList.iterator();
    while (iterator.hasNext()) {
        add(iterator.next());
    }
    return otherList.size() > 0;
}
public boolean contains(Object object) {
    for (int i = size - 1; i >= 0; i--) {
        if (list[i].equals(object)) return true;
    }
    return false;
}
public void clear() {
    for (int i = size - 1; i >= 0; i--) {
        list[i] = null;
    }
    size = 0;
    modCount++;
}
public boolean isEmpty() {
    return size == 0;
}
/**
* Returns an iterator of the elements in this list. The remove
* method is supported. Concurrent modifications (mutations) to
* the list while iterating will cause a ConcurrentModificationException
* in the subsequent <code>Iterator hasNext() or next()</code>.
* <p>
* Note that <code>Iterator.remove()</code> can be called only once
* per element, and removes the latest <code>Iterator.next()</code> 
* returned element.
*/
public Iterator iterator() {
    return new Iter();    
}
/**
* Removes the first occurance of the element if found. Returns
* true if found and removed or false if not found.
*/
public boolean remove(Object element) {
    for (int i = 0; i < size; i++) {
        if (list[i].equals(element)) {
            // Note remove() increments modCount
            remove(i + 1); // Since remove(int) is one based
            return true;
        }
    }
    return false;
}
public int size() {
    return size;
}
//---------- List Implementation ---------------------------------
public Object get(int index) {
    rangeCheck(index);
    return list[index - 1];
}
public int indexOf(Object element) {
    for (int i = 0; i < size; i++) {
        if (list[i].equals(element)) return i + 1;
    }
    return -1;
}
public Object set(int index, Object element) {
    rangeCheck(index);
    Object oldElement = list[index - 1];
    list[index - 1] = element;
    modCount++;
    return oldElement;
}
public Object add(int index, Object element) {
    if (index != size + 1) rangeCheck(index);
    // Make room for insertion
    if (size + 1 > list.length) increaseCapacity();
    // Insert
    System.arraycopy(list, index - 1, list, index, size - index + 1);
    list[index - 1] = element;
    size++;
    modCount++;
    return element;
}
public Object remove(int index) {
    rangeCheck(index);
    Object removedObject = list[index - 1]; // Convert to zero based
    
    if (index != size) {
        System.arraycopy(list, index, list, index - 1, size - index);
    }
    list[size - 1] = null;
    size--;
    modCount++;
    return removedObject;
}
public void ensureCapacity(int minCapacity) {
    if (minCapacity < list.length) return;
    Object[] oldList = list;
    list = new Object[minCapacity];
    System.arraycopy(oldList, 0, list, 0, size); 
    modCount++;
}
//---------- Public Methods --------------------------------------
public String listElements(String delimiter) {
    StringBuffer text = new StringBuffer();
    for (int i = 0; i < size; i++) {
        text.append(list[i].toString());
        if (i < size - 1) text.append(delimiter);
    }
    return text.toString();
}
public void removeAllStartingAt(int index) {
    rangeCheck(index);
    // Start at end for simplicity, speed and efficiency
    for (int i = size - 1; i >= index - 1; i--) {
        list[i] = null;
    }
    size = index - 1;
    modCount++;
}
public Object removeFirst() {
    return remove(1);
}    
public Object removeLast() {
    return remove(size);
}
//---------- Protected Methods -----------------------------------
protected void rangeCheck(int index) {
    if (index < 1 || index > size) 
        throw new IndexOutOfBoundsException
        ("All indexes must be between 1 and my size, inclusive. The index = "
        + index + " and my current size = " + size + ".");
}
protected void increaseCapacity() {
    //print(".increaseCapacity() - BEFORE, size=" + size + ", length=" + list.length);
    ensureCapacity((list.length * 3)/2 + 1);
    //print(".increaseCapacity() - After, size=" + size + ", length=" + list.length);
}
//---------- Standard --------------------------------------------
private static void print(String text) {
    System.out.println("ListOne" + text);
}
//========== Inner Classes =======================================
protected class Iter implements Iterator {

protected int expectedModCount = modCount;
protected int index = 1; // First one base index
protected int lastReturnedIndex = -1;

public boolean hasNext() {
	if (modCount != expectedModCount) throw new ConcurrentModificationException();
	return index <= size();
}
public Object next() {
	if (modCount != expectedModCount) {
	    throw new ConcurrentModificationException();
    } else if (hasNext()) {
        lastReturnedIndex = index;
        return get(index++);
    } else {
        throw new IllegalStateException("No more elements.");
    }
}
public void remove() {
    if (lastReturnedIndex < 0) {
        throw new IllegalStateException("No valid last next() index to remove.");
	} else if (modCount != expectedModCount) {
	    throw new ConcurrentModificationException();
    } else {
        ListOne.this.remove(lastReturnedIndex);
        // This reduces our current index, so:
        index--;
        // The modCount has changed, so:
        expectedModCount = modCount;
        // Allow only one remove per element
        lastReturnedIndex = -1;
    }
}

} // End inner class

} // End class