package vcb.engine.tron;

//import vcb.engine.util.EngineLib;
import vcb.gen.util.ListOne;

/**
* This represents a key value. The key is a String and the value
* is one of 6 types: int, double, boolean, String, Datatron or
* Object.
* Type specific methods are supplied for primitives. String,
* Datatron and Object are grouped into Object and type.
* String and Object values may be null.
* <p>
* The getter methods will perform automatic type transformation
* when no loss of data occurs. The most common case is when the
* value is a String, and a primitive getter is called. For example
* "false", "23" and "15.7" would be converted by isTrue(),
* getInt() and getDouble() to the proper primitives. This feature
* allows DK to be easily stored as text, parsed into a Datatron
* as String values, and be easily used by parts as intended.
* <p>
* If the String value is null then coerced getters for int, 
* double and boolean will return 0, 0 and false. This is sensible.
* <p>
* For speed do not use <code>new Entry</code>. Instead use the
* create() and release() methods, which pool unused instances.
* Once an Entry has been released it can be reused with create().
* After that its key and value must be set. Attempting to get the
* key or value if not set results in an IllegalStateException.
* <p>
* This class and the Entry pool in particular is not synchronized.
* <p>
* This class is mostly for datatron partition internal use only.
* <p>
* The complexity of this class demonstrates a weakness in Java.
* It needs to treat primitives and objects the "same".
*
* @author Jack Harich
*/
class Entry implements EntryReader {

//---------- Package Fields --------------------------------------
String key; // For faster key searches

//---------- Internal Fields -------------------------------------
protected static ListOne entryPool = new ListOne();

protected Class   type;

protected int      intValue;
protected double   doubleValue;
protected boolean  booleanValue;
protected Object   objectValue;

//---------- EntryReader Implementation --------------------------
/**
* Returns the entry in a "key=value" String representation.
*/
public String toString() {
    return key + "=" + getString();
}
/**
* Returns just the value in a String representation.
*/
public String valueToString() {
    return getString();
}
/**
* Returns true if the type is String, Datatron or Object and the
* value is null. This allows an entry to
* be parsed into an empty XML value rather than "null".
*/   // ***  or the datatron is empty
public boolean isNull() {
    if (type == String.class || type == Datatron.class || type == Object.class) {
        return objectValue == null;
    } else { // type is int, double, boolean or null
        return false; 
    }
}
/**
* Determines if the entry equals another object. Returns true if
* the object is an Entry with the equivalent key, type and value.
*/
public boolean equals(Object object) {
    if (object == null) {
        return false;
    } else if (! (object instanceof Entry) ) {
        return false;
    } else {
        Entry entry = (Entry)object;
        // Compare key
        if (! isEqualTo(entry.key, key)) return false;
        // Compare type
        if (! isEqualTo(entry.type, type)) return false;       
        // Compare value
        if (type == null) {
            return true;
        } else if (type == String.class) {
            return isEqualTo(entry.objectValue, objectValue);
            
        } else if (type == int.class) {
            return (entry.intValue == intValue);
            
        } else if (type == double.class) {
            return (entry.doubleValue == doubleValue);
                        
        } else if (type == boolean.class) {
            return (entry.booleanValue == booleanValue);
            
        } else if (type == Datatron.class) {
            return isEqualTo(entry.objectValue, objectValue);
            
        } else if (type == Object.class) {
            return isEqualTo(entry.objectValue, objectValue);
        } else {
            complain("Unknown state.");
            return false; // Never reached
        }
    }
}
/**
* Returns a shallow copy.
*/ 
public Object clone() {
    Entry entry = create();
    entry.key          = key;
    entry.type         = type;
    entry.intValue     = intValue;
    entry.doubleValue  = doubleValue;
    entry.booleanValue = booleanValue;
    entry.objectValue  = objectValue;
    return entry;
}
/**
* Returns the key, which you should never let be null. 
* The key can be null if it was never set.
*/
public String getKey() {
    if (key == null) complain("Key is null.");
    return key;
}
/**
* Returns the value type as determined by the last add or set. 
*/
public Class getType() {
    if (type == null) complain("Type is null");
    return type;
}
/**
* Returns the value as an int, performing an conversions that do
* not cause loss of data. Specifically Strings and doubles can be
* converted if they contain no fractional part. Boolean, datatron
* and Object values cannot be converted.
*/
public int getInt() {
    if (type == int.class) {
        return intValue;
    } else if (type == String.class) {
        if (objectValue == null) return 0;
        return Integer.parseInt((String)objectValue);
    } else if (type == double.class) {
        // Avoid loss of data if fractional part present        
        int possibleInt = (int)doubleValue; // Drops fraction
        if ((double)possibleInt == doubleValue) {
            return possibleInt; // No loss since int == double value
        } else {
            complain("Cannot convert double to int without loss of data.");
            return 0; // Never reached
        }
    } else { 
        complain("Cannot get boolean, datatron or Object value as int.");
        return 0; // Never reached
    }
}
/**
* Returns the value as a double, performing conversions that do
* not cause loss of data. All ints can be converted.
* Strings can be converted if they are a valid double. Boolean,
* datatron and Object values cannot be converted.
*/
public double getDouble() {
    if (type == double.class) {
        return doubleValue;
    } else if (type == int.class) {
        return (double)intValue;
    } else if (type == String.class) {
        if (objectValue == null) return 0;
        return Double.valueOf((String)objectValue).doubleValue();
    } else { 
        complain("Cannot get boolean, datatron or Object value as double.");
        return 0; // Never reached
    }
}
/**
* Returns the value as a boolean, performing conversions that do
* not cause loss of data. Int, double or String zeros are false.
* Int, double or String ones are true. A String "false" is false.
* A String "true" is true. Datatrons and Objects cannot be
* converted. All null values are inconvertable rather than 
* returning false.
*/
public boolean isTrue() {
    if (type == boolean.class) {
        return booleanValue;
    } else if (type == int.class) {
        if (intValue == 0) {
            return false;
        } else if (intValue == 1) {
            return true;
        } else {
            complain("Cannot convert int value to boolean.");
            return false; // Never reached
        }
    } else if (type == double.class) {
        if (doubleValue == 0) {
            return false;
        } else if (doubleValue == 1) {
            return true;
        } else {
            complain("Cannot convert double value to boolean.");
            return false; // Never reached
        }
    } else if (type == String.class) {
        if ("false".equals((String)objectValue) || objectValue == null) {
            return false;
        } else if ("true".equals((String)objectValue)) {
            return true;
        } else {
            complain("Cannot convert String value to boolean.");
            return false; // Never reached
        }        
    } else if (type == Datatron.class) {
        complain("Cannot convert Datatron value to boolean.");
        return false; // Never reached
    } else if (type == Object.class) {
        complain("Cannot convert Object value to boolean.");
        return false; // Never reached
    } else { // type is null
        complain("Unknown type.");
        return false; // Never reached
    }
}
/**
* Returns the String representation of the value for all types.
*/
public String getString() {
    if (type == String.class) {
        return (String)objectValue;
    } else if (type == int.class) {
        return String.valueOf(intValue);
    } else if (type == double.class) {
        return String.valueOf(doubleValue);
    } else if (type == boolean.class) {
        return String.valueOf(booleanValue);
    } else if (type == Datatron.class || type == Object.class) {
        if (objectValue == null) {
            return null;
        } else {
            return objectValue.toString();    
        }
    } else { // type is null
        // Avoid circular call - complain() calls toString()
        throw new IllegalStateException("Cannot get String " +
            "because type is unknown for Key = " + key);
    }
}
/**
* Returns a String representation of the value suitable for
* HashCode use. If the type is a Datatron or Object then "Complex"
* is returned. Otherwise the normal String value is returned.
*/
public String getHashCodeStringForValue() {
    if (type == Datatron.class || type == Object.class) {
        return "Complex"; // Prevents going deep and slow
    } else {
        return getString();
    }
}
/**
* Returns the value as a Datatron, performing conversions that
* make sense. If an Object is a Datatron it's returned. If it's a
* null String null is returned.
*/
public Datatron getDatatron() {
    if (type == Datatron.class) {
        return (Datatron)objectValue;
    } else if (type == Object.class && objectValue instanceof Datatron) {
        return (Datatron)objectValue;
    } else if (type == String.class && objectValue == null) {
        // This would be inconsistent, so don't do.
        //***return Datatron.create();
        return null;
    } else {
        complain("Cannot convert int, double, boolean, " +
            "non-Datatron Object or no type to Datatron.");
        return null; // Never reached    
    }
}
/**
* Returns the value as an Object, performing conversions that
* make sense. The only ones that do are String and Datatron.
*/
public Object getObject() {
    if (type == String.class || type == Datatron.class || type == Object.class) {
        return objectValue;
    } else { // type is int, double, boolean or null
        complain("Cannot convert int, double, boolean or no type to Object.");
        return null; // Never reached
    }
}
//---------- Public Methods --------------------------------------
//----- Pool
/**
* Creates an Entry from the Entry pool or a new one if necessary.
* The Entry will contain no key or value.
*/
static Entry create() {
    if (entryPool.isEmpty()) {
        return new Entry();
    } else {
        return (Entry)entryPool.removeLast();
    }
}
/**
* Releases this Entry from use and returns it to the pool. This
* should be done when the instance is no longer needed. Internal
* object references will be nulled out to prevent the Entry from
* preventing garbage collection of the key or value.
*/
public void release() {
    // Reset before putting in pool
    key           = null;
    type          = null;
    objectValue   = null;
    // Primitives are not reset since values are controlled
    // Return to pool
    entryPool.add(this);
}
//----- Type mutators
/**
* Sets the value to an int, overriding any previous value of 
* any type.
*/
public void setInt(int value) {
    type = int.class;
    intValue = value;
}
/**
* Sets the value to a double, overriding any previous value of 
* any type.
*/
public void setDouble(double value) {
    type = double.class;
    doubleValue = value;
}
/**
* Sets the value to a boolean, overriding any previous value of 
* any type.
*/
public void setTrue(boolean value) {
    type = boolean.class;
    booleanValue = value;
}
/**
* Sets the value to an Object and the type to the supplied type,
* overriding any previous value of any type. The type should be
* String.class, Datatron.class or Object.class. This allows
* using the object value slot for multiple object types.
*/
public void setObject(Object value, Class type) {
    if (type == String.class || type == Datatron.class || type == Object.class) {
        objectValue = value;
        this.type = type;
    } else {
        complain("Invalid type: " + type);
    }
}
//---------- Package Methods -------------------------------------
/**
* Sets the entry key, which should follow standard Java name
* conventions. It should contain only letters, numbers or
* underscores and begin with a letter. Absolutely do not embed
* spaces or periods. This is a required field and cannot be null.
*/
void setKey(String key) {
    if (key == null) complain("Key cannot be null.");
    this.key = key;
}
//---------- Protected Methods -----------------------------------
protected boolean isEqualTo(Object object1, Object object2) {
    if (object1 == null && object2 == null) {
        return true; 
    } else if (object1 == null || object2 == null) {
        return false;
    } else {
        return object1.equals(object2);
    }
}
protected void complain(String text) {
    throw new IllegalStateException(text + " Entry=" + 
        toString() + ", Type=" + type);
}
//---------- Standard --------------------------------------------
private static void print(String text) {
    System.out.println("Entry" + text);
}

} // End class