package vcb.engine.tron;

import java.util.Iterator;
import java.util.NoSuchElementException;

/**
* This provides a type specific iterator for a Datatron and the
* usual Iterator. The benefits are more concise and clear use
* of the entries in the iterator. Since this class implements
* Iterator it can also be used in general purpose ways, such as
* in collection utilities that expect an Iterator.
* <p>
* The element returned in <code>next()</code> or 
* <code>nextEntry()</code> is an EntryReader.
* @see vcb.engine.tron.EntryReader
*
* @author Jack Harich
*/
public class DatatronIterator implements Iterator {

//---------- Internal Fields -------------------------------------
protected int     index; // Start at zero
protected int     size;
protected Entry[] entries;

//---------- Initialization --------------------------------------
/**
* Constructs a new instance with the supplied entries and size.
*/
DatatronIterator(Entry[] entries, int size) {
    this.entries = entries;
    this.size    = size;
}
//---------- Iterator Implementation -----------------------------
/**
* Determines if the iterator has more elements as determined by
* the current index and size of elements.
*/
public boolean hasNext() {
    return (index < size ? true : false);
}
/**
* Returns the next Entry and advances the current index.
*/
public Object next() {
    if (index < size) {
        return entries[index++];
    } else {
        throw new NoSuchElementException("No more entries.");
    }
}
/**
* Not supported so throws an UnsupportedOperationException.
* It's possible we will support this in the future.
*/
public void remove() { // Perhaps later allow this
    throw new UnsupportedOperationException("Remove not allowed.");
}
//---------- Public Methods --------------------------------------
/**
* Returns the next EntryReader and advances the current index.
*/
public EntryReader nextEntry() {
    return (EntryReader)next();
}
/**
* Returns the next Datatron in the iteration, which requires the
* next element to be an entry that is a Datatron type. This is
* very handy for Datatron tree use.
*/
public Datatron nextDatatron() {
    return ((EntryReader)next()).getDatatron();
}
//---------- Standard --------------------------------------------
private static void print(String text) {
    System.out.println("DatatronIterator" + text);
}

} // End class