package jsl.fw1.engine;

import vcb.inv.core.ClassResourceReader;
import vcb.engine.tron.Datatron;
import vcb.engine.tron.DatatronIterator;
import vcb.engine.tron.Parser;
import java.util.HashMap;
import java.util.Iterator;

/**
* A container contains zero or more other nodes.
*
* @author Jack Harich
*/
public class Container extends Node {

//---------- Internal Fields -------------------------------------
protected Datatron myDK;
protected boolean  isInitialized;

// Key = Node name, Object = Node
protected HashMap nodes = new HashMap();

// Shared
protected static ClassResourceReader reader = new ClassResourceReader();
protected static Parser parser = new Parser();

protected static final String DK_FILE_NAME = "_Container.dk";

//---------- Initialization --------------------------------------
protected Container() { } // Enforce use of createRoot(), etc.

//---------- Superclass Overrides --------------------------------
public boolean isDKDriven() {
    return true;
}
public void setDK(Datatron dk) {
    myDK = dk;
    //print(".setDK() - The dk is:\n" + dk.toXML("dk"));
}
//---------- Public Methods --------------------------------------
/**
* Returns the root container for a system. The markerClass must be
* a "marker" class located in the root directory of the system.
*/
public static Container createRoot(String markerClass) {
    String text = reader.readText(markerClass, DK_FILE_NAME);
    Datatron dk = parser.parseXMLtoDatatron(text);

    Container root = new Container();
    root.setName("Root");
    root.setDK(dk);
    return root;
}
/**
* Adds a node to the container. The node's name must not already
* be present in the container.
*/
public void addNode(Node node) {
    if (nodes.containsKey(node.getName())) throw new IllegalStateException
        ("The node name " + node.getName() + " is already present.");
    nodes.put(node.getName(), node);    
}
/**
* Returns an iteration of all nodes in no particular order.
*/
public Iterator iterateNodes() {
    checkDeferredInit();
    return nodes.values().iterator();
}
/**
* Returns the node with the name. If not found an IllegalStateException
* is thrown.
*/
public Node getNode(String name) {
    checkDeferredInit();
    if (nodes.containsKey(name)) {
        return (Node)nodes.get(name);
    } else {
        throw new IllegalStateException("The name " + name + " is not present.");
    }
}
//---------- Protected Methods -----------------------------------
protected void checkDeferredInit() {
    if (isInitialized) return;
    
    isInitialized = true;
    DatatronIterator nodes = myDK.getDatatron("Nodes").iterateEntries();
    while (nodes.hasNext()) {
        Datatron nodeDK = nodes.nextDatatron();
        String name = nodeDK.getString("Name");
        String type = nodeDK.getString("Type");
        Node node = createNode(name, type);
        addNode(node);
    }
}
protected Node createNode(String name, String type) {
    if (nodes.containsKey(name)) throw new IllegalStateException
        ("The name " + name + " already exists.");
    Node node = null;
    if (type.equals("Container")) {
        node = new Container();
    } else {
        try {
            node = (Node)Class.forName(type).newInstance();
        } catch(Exception ex) {
            ex.printStackTrace();
            throw new RuntimeException("Cannot create type " + type);
        }
    }
    node.setName(name);
    node.setParent(this);
    if (node.isDKDriven()) provideNodeDK(node);
    return node;
}
protected void provideNodeDK(Node node) {
    print(".provideNodeDK() - Not yet implemented");
}
//---------- Standard --------------------------------------------
private static void print(String text) {
    System.out.println("Container" + text);
}

} // End class