package vcb.engine.tron;

import vcb.gen.util.ListOne;
import java.util.Iterator;

/**
* A Datatron carries data from a part output to a part input.
* It's designed to carry mostly domain neutral data, which
* preserves Anonymous Collaboration among parts. A datatron has
* zero of more key values. Keys are
* Strings. Keys may be duplicates and are ordered. The values can
* be int, double, boolean, String, Datatron or Object. 
* A key value is an Entry internally.
* <p>
* The benefits of the 6 datatypes are strong typing, speed and 
* ease of use with primitives, clear trees of datatrons for 
* key.key.key access, flexibility by inclusion of Object, easy
* datatype coercion and good XML (DK) representation.
* <p>
* Datatrons can be nested by making a value a Datatron.
* Nesting gives a tree structure that is very useful. 
* The most common tree use is for a part's DK.
* <p>
* For speed a part should reuse a (or several) datatron rather 
* than creating a new one for each send. To reuse a datatron call 
* <code>clear()</code>. When the part ends its life it should 
* release the datatron if possible, such as in its finalize().
* <p>
* Note the keys are maintained in order. This allows the order of
* keys to fully represent DK, such as the way HTML works. It also
* allows reversing a Datatron to XML with no loss of meaning.
* It also allows duplicate keys for situations where they are
* needed, particularly list collection types. This solves the
* awkwardness of a separate collection for lists in datatrons.
* <p>
* For a map style datatron use only "set" methods. For a list style
* use "add" methods. To allow morphing of a map to a list if a
* duplicate key is encountered use "add" methods, such as when
* parsing XML into a datatron. Note that even if add methods have
* been used the collection may still have unique keys.
* <p>
* <b>Note that once the isList property goes true, gets and sets
* will throw an exception.</b> 
* <p>
* Typically a datatron only has a few keys and so the linear 
* search necessitated by ordered and duplicate keys allowed
* is not slow. We expect most of the large datatrons are for DK,
* which is often used only once by a part on startup, and so
* key retrevial speed is a minor issue. If speed does become an
* issue we can add an underlying Map.
* <p>
* Parts collaborate solely via datatrons. The two main use cases 
* are a part's DK and a part making the equivalent of a method
* call to another part.
* <p>
* DK is parsed from XML and can be quite complex. The parser 
* cannot know if a collection of keys is unique or not while 
* parsing, until it's run through all a container tag's children
* (keys). It also doesn't know if order is important. Thus the 
* parser simply adds the key and values to the datatron, which 
* happily maintains the key values in the order added and allows 
* access by key or iteration. The part using the datatron "knows" 
* whether a collection is unique keys or a list, and so uses gets
* or an iteration as appropriate. This was a sticky mess of 
* conflicting use cases to resolve. I'm pleased with the design 
* and implementation. 
* <p>
* <b>A Datatron class is conceptually lightweight</b>. It has a lot
* of public methods, about 44 so far, but 24 of them follow a pattern. 
* The pattern is addType, setType, getType and getTypeDefault, 
* where "Type" is Int, Double, True, String, Datatron or Object. 
* (6 types x 4 accessor methods per type = 24 methods) For better
* readability the getters for True use an "is" prefix. The 
* following applies to each key value accessor pattern:
* <p>
* <b>void addType(String key, Type value)</b> -
* Adds the key value at the end of the collection,
* regardless of whether the key is already in the collection.
* This means the collection may now have duplicate keys and so
* should be treated as a list. For example this is used when
* creating a Datatron from XML.
* <p>
* <b>void setType(String key, Type value)</b> -
* Sets the value for the key. If the key is new the key value is
* added at the end of the collection. If the key already exists
* the previous value is overwritten, regardless of previous value
* type. This is the method normally used by parts
* when collaborating with datatrons for unique keys.
* <p>
* <b>Type getType(String key)</b> -
* Returns the value for the first entry with the key. Data
* conversion is attempted if the underlying datatype is not the 
* Type. Throws an IllegalStateException if the key is not found
* for primitives or returns null for String, Datatron or Object.
* <p>
* <b>Type getTypeDefault(String key, Type defaultValue)</b> -
* Returns the value for the first entry with the key or the
* supplied default if not found. If found data conversion is 
* attempted if the underlying datatype is not the Type.
* <p>
* Later we will support key.key.key style accessors, which is why
* we forbid periods in keys. Clone and deepClone is not yet 
* implemented.
* <p>
* This is such a crucial class with so many methods we offer a
* condensed list here for quick reference:
* <pre>
*   General Use
*     Pool of Datatrons for speed
*       public static Datatron create() - From pool
*       public void release() - Return to pool
*     Superclass Overrides
*       public String toString() - Returns tree representation
*       public boolean equals(Object object) - Standard use
*       public int hashCode() - For use as a key in a map
*     General
*       public boolean isList() - True if has dupe keys
*       public int size() - Number of entries
*       public EntryReader getEntryAt(int index) - If list, base one
*       DatatronIterator iterateEntries() - Iteration of EntryReader
*       public boolean isEmpty() - True if zero entries
*       public boolean containsKey(String key) - True if has key
*       public Class getValueType(String key) - For inspection
*       public boolean isEquivalentMap(Datatron dt)
*       public addAll(Datatron dt) - Adds all entries in dt
*       public Datatron sortKeys() - For use as a map key
*       public String toXML(String rootTag) - For general use
*     Entry Work
*       public EntryReader getEntry(String key)
*       public EntryReader getFirstEntry()
*       public void setEntry(String key, EntryReader entry)
*     Removers
*       public Datatron clear() - Empties collection
*       public boolean removeKey(String key) - True if removed
*   Data Accessors
*      Type int
*        public void addInt(String key, int value)
*        public void setInt(String key, int value)
*        public int getInt(String key)
*        public int getIntDefault(String key, int defaultValue)
*      Type double
*        public void addDouble(String key, double value)
*        public void setDouble(String key, double value)
*        public double getDouble(String key)
*        public double getDoubleDefault(String key, double defaultValue)
*      Type boolean
*        public void addTrue(String key, boolean value)
*        public void setTrue(String key, boolean value)
*        public boolean isTrue(String key)
*        public boolean isTrueDefault(String key, boolean defaultValue)
*      Type String
*        public void addString(String key, String value)
*        public void setString(String key, String value)
*        public String getString(String key)
*        public String getStringDefault(String key, String defaultValue)
*      Type Datatron
*        public void addDatatron(String key, Datatron value)
*        public void setDatatron(String key, Datatron value)
*        public Datatron getDatatron(String key)
*        public Datatron getDatatronDefault(String key, Datatron defaultValue)
*        public Datatron readDatatron(String key)
*      Type Object
*        public void addObject(String key, Object value)
*        public Datatron setObject(String key, Object value)
*        public Object getObject(String key)
*        public Object getObjectDefault(String key, Object defaultValue)
* </pre>
* @author Jack Harich
*/
public class Datatron {

//---------- Internal Fields -------------------------------------
// Contains Datatrons for reuse
protected static ListOne myPool = new ListOne();

protected EntryMapList entries = new EntryMapList();

//---------- Initialization --------------------------------------
protected Datatron() {
    // Must use create()
}
//---------- Object Overrides ------------------------------------
/**
* Returns a String representation of the collection in a tree
* format, since often the contents are a tree.
*/
public String toString() {
    return DatatronLib.toString(entries);
}
public void finalize() {
    release();
}
public boolean equals(Object object) {
    if (object == null) {
        return false;
    } else if (! (object instanceof Datatron) ) {
        return false;
    } else {
        // Compare entries
        return ((Datatron)object).entries.equals(entries);
    }
}
public int hashCode() {
    return entries.hashCode();
}
//---------- Public Methods --------------------------------------
//----- Pool
/**
* Creates a Datatron from the Datatron pool or a new one if 
* necessary. The Datatron will contain no key values.
*/
public static Datatron create() {
    if (myPool.isEmpty()) {
        return new Datatron();
    } else {
        return (Datatron)myPool.removeLast();
    }
}
/**
* Releases this Datatron from use and returns it to the pool. This
* should be done when the instance is no longer needed. Internal
* object references will be nulled out to prevent the Datatron from
* preventing garbage collection of the name or key values.
*/
public void release() {
    // Reset before putting in pool
    clear();
    // Return to pool
    myPool.add(this);
}
//----- General
/**
* Determines whether this is a list or not, in other
* words whether duplicate keys are present. This property
* goes true when the first duplicate key is added.
* It goes false when the collection is emptied or the last
* duplicate key is removed.
* <p>
* <b>Note that once the isList property goes true, gets and sets
* will throw an exception.</b> 
*/
public boolean isList() {
    return entries.isList();
}
/**
* Returns the number of key values (entries) in the Datatron,
* which is zero or more.
*/
public int size() {
    return entries.size();
}
/**
* Returns the entry reader at the <b>base one</b> index. 
* If the index is out of bounds throws an IndexOutOfBoundsException.
* Note this can be used for isList equals true or false.
*/
public EntryReader getEntryAt(int index) {
    return entries.getEntryAt(index);   
}
/**
* Returns an interation of EntryReader instances for read only
* access to each entry in the collection. Remove is not allowed.
* @see vcb.engine.tron.EntryReader
* @see vcb.engine.tron.DatatronIterator
*/
public DatatronIterator iterateEntries() {
    return entries.iterateEntries();
}
/**
* Determines if the Datatron is empty.
*/ 
public boolean isEmpty() {
    return entries.isEmpty();
}
/**
* Determines if the Datatron contains the key.
*/
public boolean containsKey(String key) {
    return entries.containsKey(key);
}
/**
* Returns the value type for the key or null if not found.
*/
public Class getValueType(String key) {
    Entry entry = entries.getEntry(key);
    return (entry == null ? null : entry.getType());
}
/**
* Determines whether each key in this class has a value that's
* the same as that key's value in the supplied Datatron.
* This is useful in cases where
* entry order can be different but map content can be the same.
* This method is not useful for duplicate keys, because
* that is a list, not a map.
*/
public boolean isEquivalentMap(Datatron dt) {
    return entries.isEquivalentMap(dt.entries);
}
/**
* Adds all entries in the supplied datatron to this datatron.
* Note these are added, not set. The added entries are cloned.
* The object, if any, in the entries in not cloned.
* Returns this datatron.
*/
public Datatron addAll(Datatron dt) {
    entries.addAll(dt.entries);
    return this;
}
/**
* Sorts the keys into alphabetical order. This must be done before
* an instance of this class can be reliably used as a storage or
* getter key in a map, because the method hashCode() returns a
* value dependent on the concatenation of all keys and values. We 
* see no other way to use this class reliably as a key.
* <p>
* Returns this datatron for easy code use.
*/
public Datatron sortKeys() {
    entries.sortKeys();
    return this;
}
/**
* Returns the datatron in XML format, using the rootTag. Note
* that this works only for datatrons with no objects other than
* datatrons or strings, because a complex object has no natural
* XML representation. If the rootTag is null then the xml is not
* wrapped in a root tag.
*/
public String toXML(String rootTag) {
    return DatatronLib.toXML(entries, rootTag);
}
//----- Entry Work
/**
* Returns the EntryReader for the key or null if not found.
*/
public EntryReader getEntry(String key) {
    return (EntryReader)entries.getEntry(key);
}
/**
* Returns the first EntryReader or null if none. One use for this
* is getting the entry in a datatron with only one entry.
*/
public EntryReader getFirstEntry() {
    return (EntryReader)entries.getFirstEntry();
}
/** 
* Sets the entry for the key. This method and
* <code>getEntryReader()</code> allow entries to be transferred
* from one datatron to another. This method should be used 
* sparingly. For safety the entry is cloned and the entry's key is
* set using the supplied key.
* Throws an IllegalStateException if isList is true.
*/
public void setEntry(String key, EntryReader entry) {
    Entry newEntry = (Entry)entry.clone();
    newEntry.setKey(key);
    entries.setEntry(newEntry);
}
//----- Removers
/**
* Removes all key values and returns this datatron for ease of use.
*/
public Datatron clear() {
    entries.clear();
    return this;
}
/**
* Removes the first value for the key, returning true if removed 
* or false if the key was not found.
*/
// Possibly removeEntry(EntryReader) later
public boolean removeKey(String key) {
    Entry entry = entries.removeKey(key);
    return (entry != null ? true : false);
}
//----- int
/**
* The int adder. See class doc for further detail.
*/
public void addInt(String key, int value) {
    Entry entry = Entry.create();
    entry.setKey(key);
    entry.setInt(value);
    entries.addEntry(entry);
}
/**
* The int setter. See class doc for further detail.
*/
public void setInt(String key, int value) {
    Entry entry = entries.getEntry(key);
    if (entry != null) {
        entry.setInt(value);
    } else {
        entry = Entry.create();
        entry.setKey(key);
        entry.setInt(value);
        entries.setNonDuplicateKeyEntry(entry);
    }
}
/**
* The int getter. See class doc for further detail.
*/
public int getInt(String key) {
    Entry entry = entries.getEntry(key);
    if (entry == null) complain("Key '" + key + "' not found.");
    return entry.getInt();
}
/**
* The int getter with default. See class doc for further detail.
*/
public int getIntDefault(String key, int defaultValue) {
    Entry entry = entries.getEntry(key);
    if (entry == null) {
        return defaultValue;
    } else {
        return entry.getInt();    
    }
}
//----- double
/**
* The double adder. See class doc for further detail.
*/
public void addDouble(String key, double value) {
    Entry entry = Entry.create();
    entry.setKey(key);
    entry.setDouble(value);
    entries.addEntry(entry);
}
/**
* The double setter. See class doc for further detail.
*/
public void setDouble(String key, double value) {
    Entry entry = entries.getEntry(key);
    if (entry != null) {
        entry.setDouble(value);
    } else {
        entry = Entry.create();
        entry.setKey(key);
        entry.setDouble(value);
        entries.setNonDuplicateKeyEntry(entry);
    }
}
/**
* The double getter. See class doc for further detail.
*/
public double getDouble(String key) {
    Entry entry = entries.getEntry(key);
    if (entry == null) complain("Key '" + key + "' not found.");
    return entry.getDouble();
}
/**
* The double getter with default. See class doc for further detail.
*/
public double getDoubleDefault(String key, double defaultValue) {
    Entry entry = entries.getEntry(key);
    if (entry == null) {
        return defaultValue;
    } else {
        return entry.getDouble();    
    }
}
//----- boolean
/**
* The boolean adder. See class doc for further detail.
*/
public void addTrue(String key, boolean value) {
    Entry entry = Entry.create();
    entry.setKey(key);
    entry.setTrue(value);
    entries.addEntry(entry);
}
/**
* The boolean setter. See class doc for further detail.
*/
public void setTrue(String key, boolean value) {
    Entry entry = entries.getEntry(key);
    if (entry != null) {
        entry.setTrue(value);
    } else {
        entry = Entry.create();
        entry.setKey(key);
        entry.setTrue(value);
        entries.setNonDuplicateKeyEntry(entry);
    }
}
/**
* The boolean getter. See class doc for further detail.
*/
public boolean isTrue(String key) {
    Entry entry = entries.getEntry(key);
    if (entry == null) complain("Key '" + key + "' not found.");
    return entry.isTrue();
}
/**
* The boolean getter with default. See class doc for further detail.
*/
public boolean isTrueDefault(String key, boolean defaultValue) {
    Entry entry = entries.getEntry(key);
    if (entry == null) {
        return defaultValue;
    } else {
        return entry.isTrue();    
    }
}
//----- String
/**
* The String adder. See class doc for further detail.
*/
public void addString(String key, String value) {
    addEntryObject(key, value, String.class);
}
/**
* The String setter. See class doc for further detail.
*/
public void setString(String key, String value) {
    setEntryObject(key, value, String.class);
}
/**
* The String getter. See class doc for further detail.
*/
public String getString(String key) {
    Entry entry = entries.getEntry(key);
    if (entry == null) {
        return null; // Not found
    } else {
        return entry.getString(); // Performs conversion
    }
}
/**
* The String getter with default. See class doc for further detail.
*/
public String getStringDefault(String key, String defaultValue) {
    String value = getString(key);
    return (value == null ? defaultValue : value);
}
//----- Datatron
/**
* The Datatron adder. See class doc for further detail.
*/
public void addDatatron(String key, Datatron value) {
    addEntryObject(key, value, Datatron.class);
}
/**
* The Datatron setter. See class doc for further detail.
*/
public void setDatatron(String key, Datatron value) {
    setEntryObject(key, value, Datatron.class);
}
/**
* The Datatron getter. See class doc for further detail.
*/
public Datatron getDatatron(String key) {
    Entry entry = entries.getEntry(key);
    if (entry == null) {
        return null;
    } else {
        return entry.getDatatron(); // Performs conversion
    }
}
/**
* The Datatron getter with default. See class doc for further detail.
*/
public Datatron getDatatronDefault(String key, Datatron defaultValue) {
    Datatron value = getDatatron(key);
    return (value == null ? defaultValue : value);
}
/**
* Reads the datatron using the key and returns it. If not found a
* new datatron with the key is set by calling 
* <code>setDatatron(...)</code>, so the datatron used can reliabily
* be mutated by mutating the returned datatron. This is useful for
* general structured storage purposes, so that the client need not
* worry whether a <code>getDatatron(String key)</code> returned a
* datatron that was in this datatron or not. Bugs are reduced. :-)
* <p>
* In the case that the entry value is null and the entry type
* is String or Object, the entry type is changed to Datatron.
* Null Strings can occur when a parser parses an empty tag to a
* null String as a default, which is common.
*/
public Datatron readDatatron(String key) {
    Datatron dt = null;
    Entry entry = entries.getEntry(key);
    if (entry == null) {
        dt = create();
        setDatatron(key, dt);
    } else if (entry.getType() == Datatron.class) {
        dt = entry.getDatatron();
    } else if (entry.getType() == Object.class && 
            entry.getObject() instanceof Datatron) {
        dt = (Datatron)entry.getObject();
        setDatatron(key, dt);
    } else if (entry.getType() == String.class && 
            entry.getString() == null) {
        dt = create();
        setDatatron(key, dt);
    } else {
        throw new IllegalStateException("The value for key '" + key
        + "' is not a Datatron and not a null String.");
    }
    return dt;
}

//----- Object
/**
* The Object adder. See class doc for further detail.
*/
public void addObject(String key, Object value) {
    addEntryObject(key, value, Object.class);
}
/**
* The Object setter. See class doc for further detail.
*/
public Datatron setObject(String key, Object value) {
    setEntryObject(key, value, Object.class);
    return this;
}
/**
* The Object getter. Returns null if not found. 
* See class doc for further detail.
*/
public Object getObject(String key) {
    Entry entry = entries.getEntry(key);
    if (entry == null) {
        return null; // Not found
    } else {
        return entry.getObject(); // Performs conversion
    }
}
/**
* The Object getter with default. See class doc for further detail.
*/
public Object getObjectDefault(String key, Object defaultValue) {
    Object value = getObject(key);
    return (value == null ? defaultValue : value);
}
//---------- Package Methods --------------------------------------
EntryMapList getEntries() { // We like to avoid these, but....
    return entries;
}
//---------- Protected Methods -----------------------------------
// String, Datatron or Object accessors
protected void addEntryObject(String key, Object value, Class type) {
    Entry entry = Entry.create();
    entry.setKey(key);
    entry.setObject(value, type);
    entries.addEntry(entry);
}
protected void setEntryObject(String key, Object value, Class type) {
    Entry entry = entries.getEntry(key);
    if (entry != null) {
        entry.setObject(value, type);
    } else {
        entry = Entry.create();
        entry.setKey(key);
        entry.setObject(value, type);
        entries.setNonDuplicateKeyEntry(entry);
    }
}
//----- Other
protected void complain(String text) {
    // We shall see if toString() is too verbose, maybe just name
    throw new IllegalStateException(text + " Datatron=" + 
        toString() );
}
//---------- Standard --------------------------------------------
private static void print(String text) {
    System.out.println("Datatron" + text);
}

} // End class