package org.jcon.core.proto.c2.std;

import org.jcon.core.proto.c2.Cell;
import org.jcon.core.proto.c2.Node;
import org.jcon.core.proto.c2.PartUser;
import org.jcon.util.minor.KeyedVector;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

/**
* This class is the standard Cell. See interface for doc.
* <p>
* Unfortunately this class has the additional responsibility
* of handling the PartUser optional interface. This may be
* done differently in different cores, especially if a
* cell lifecycle part is present.
*
* @author Jack Harich
*/

public class CellStd implements Cell {

//---------- Protected Fields ----------------------------
protected KeyedVector nodes = new KeyedVector();
protected Node        myNode;

// Key = partName, Object = Vector of PartUsers
protected Hashtable partNeeders = new Hashtable();

//---------- Cell Implementation -------------------------
public void insertNodeAt(Node node, int index) {
    duplicateNameCheck(node);
    nodes.insertElementAt(node, node.getName(), index); 
    initPart(node.getPart(), node.getName());
}
public void appendNode(Node node) {
    duplicateNameCheck(node);
    nodes.addElement(node, node.getName());
    initPart(node.getPart(), node.getName());    
}
public void clear() {
    nodes.removeAll();
}
public Node removeNode(Node node) {
    return (Node)nodes.removeElement(node.getName());
}
public int getNodeCount() {
    return nodes.size();
}
public Node getNodeAt(int index) {
    return (Node)nodes.getElementAt(index);
}
public Enumeration getNodes() {
    return nodes.elements();
}
public Node getNode(String name) {
    return (Node)nodes.getElementForKey(name);
}
public Node getCellNode() {
    return myNode;
}
public Object getPart(String nodeName) {
    return ((Node)nodes.getElementForKey(nodeName)).getPart();
}
public Object findPart(String partName) {
    // Special case 
    if (partName.equals("MyCell")) return this;
    
    // Perform 4 level search
    Node node = getNode(partName);
    
    // Case 1 - Found in this container
    if (node != null) return node.getPart();
    
    // Search upward in each cell until root encountered
    Node upwardNode = myNode.getParentNode();
    while (upwardNode != null) {
        Cell upwardCell = (Cell)upwardNode.getPart();
        node = upwardCell.getNode(partName);
        if (node != null) break;
        upwardNode = upwardNode.getParentNode();
    }
    // Case 2 - Never found upwardly
    if (node == null) {
        throw new IllegalStateException("Part '" + partName +
        "' not found in this container or above.");
    }
    // Case 3 - Found upwardly, is inheritable part
    if (node.isInheritable()) {
        return node.getPart();
    }
    // Case 4 - Found upwardly, not inheritable part
    throw new IllegalStateException("Part '" + partName +
    "' found upwardly but is not an inheritable part.");
}
//----- NodeUser Implementation
public void setNode(Node node) {
    myNode = node;
}
//----------- Replicable Implementation ------------------
public Object replicate() {
    // No DK so this is easy
    return new CellStd();
}
//---------- Protected Methods ---------------------------
protected void duplicateNameCheck(Node node) {

    if (node.getName() == null) {
        throw new IllegalStateException("Cannot handle"
        + " nodes with a null name.");
    }

    if (nodes.containsKey(node.getName())) {
        throw new IllegalStateException("The node named '"
        + node.getName() + "' is already in this cell.");
    }
}
/**
* Handles the PartUser optional interface. This is done in
* a manner that allows parts to be added in any order.
* If a PartUser's needs are never met, no exception or
* such is thrown, and the part will probably error out
* when it needs the missing part. This can be corrected
* in a cell lifecycle approach.
*/
protected void initPart(Object part, String partName) {
    // If PartUser, provide parts needed
    if (part instanceof PartUser) {
        PartUser partUser = (PartUser)part;
        String[] names = partUser.getPartNeeds();
        
        // Supply part for each need
        for (int i = 0; i < names.length; i++) {
            supplyPart(names[i], partUser);
        }    
    }    
    // Check partNeeders for part arrival
    Vector needers = (Vector)partNeeders.get(partName);
    if (needers == null) return; // None awaiting this part
    
    // Part has arrived and has needers
    for (int i = 0; i < needers.size(); i++) {
        PartUser partUser = (PartUser)needers.elementAt(i);
        partUser.setPart(partName, part);
    }
    partNeeders.remove(partName);
}
// Supplies named part to user. If not found, adds user to
// partNeeders for partName.
protected void supplyPart(String partName, PartUser user) {
    Object partNeeded;
    try {
        partNeeded = findPart(partName);
        user.setPart(partName, partNeeded);
        
    } catch(Exception ex) {
        // Not found so add to partNeeders
        Vector needers = (Vector)partNeeders.get(partName);
        if (needers == null) {
            needers = new Vector();
            partNeeders.put(partName, needers);
        }
        needers.addElement(user);
    } 
}
//---------- Private Methods -----------------------------
//--- Std
private static void print(String text) {
    System.out.println("CellStd" + text);
}


} // End class