package org.jcon.util.setting;

import org.jcon.ba.system.BeanActionClose;
import org.jcon.ba.system.BeanActionStart;
import org.jcon.util.GenLib;
import org.jcon.util.msg.Message;
import org.jcon.util.msg.MessageListener;
import org.jcon.util.service.ContainerServices;
import org.jcon.util.service.ContainerServicesUser;
import java.awt.*;
import java.awt.event.*;


/**
 * This class allows the user to edit a Setting's local
 * value. This means the change affects only them. For
 * example a user may edit the CurrentFiscalYear Setting
 * to filter a task for that year.
 * <p>
 * Later we will support the ablity to edit multiple
 * setting on a single window. We are starting simple.
 *
 * @author Jack Harich
 */
// Combined view and controller. Split if gets complex.
// java org.jcon.util.setting.EditLocalSetting
public class EditLocalSetting implements MessageListener,
    BeanActionStart, BeanActionClose, 
    ContainerServicesUser {

//---------- Private Fields ------------------------------
// Properties
private String            settingName;
private String            startMessageName;
private ContainerServices containerServices;
// Working
private EditLocalSettingView view = new EditLocalSettingView();
private SettingStore settingStore;

//---------- Initialization ------------------------------
public static void main(String args[]) {
     new EditLocalSetting().runUnitTest();   
}    
//---------- MessageListener Implementation --------------
public void processMessage(Message message) {
    String name = message.getName();
    //print(".processMessage() - name = '" + name + "'");
    if (name.equals(startMessageName)) start();
}
public String[] loadMessageInterests() {
    if (startMessageName == null) {
        return new String[] {};
    } else {
        return new String[] {startMessageName};
    }
}
//---------- BeanActionStart Implementation --------------
// We can start this way or by a message
public void start() {
    view.setController(this);
    view.init();
    try { // So unit test works
        settingStore = (SettingStore)
            containerServices.getService("SettingStore");
        // Set view values
        view.setEditLabel(settingStore.getSettingDef(settingName).getDisplayName());
        view.setEditValue(settingStore.getLocalValue(settingName));
        view.setLocalValue(settingStore.getLocalValue(settingName));
        view.setActualValue(settingStore.getActualValue(settingName));
    } catch(Exception ex) {
         ex.printStackTrace();   
    }
    // Done
    view.setVisible(true);
}
//---------- BeanActionClose Implementation --------------
public String canClose() {
    return null; // Yes *** later check dirty
}
// Revert to restartable state
public void close() {
    view.close();
}    
//---------- ContainerServicesUser Implementation --------
public void setContainerServices(ContainerServices services) {
    containerServices = services;
}
public String[] getContainerServicesInterests() {
    return new String[] {"SettingStore"};    
}
//---------- Properties ----------------------------------
//----- title
public void setTitle(String title) {
     view.setTitle(title);
}    
public String getTitle() {
     return view.getTitle();;   
}
//----- settingName
public void setSettingName(String settingName) {
    this.settingName = settingName;
    view.setSettingName(settingName); 
}    
public String getSettingName() {
     return settingName;
}    
//----- fieldSize
public void setFieldSize(int fieldSize) {
    view.setFieldSize(fieldSize); 
}
public int getFieldSize() {
    return view.getFieldSize();   
}    
//----- startMessageName
public void setStartMessageName(String startMessageName) {
     this.startMessageName = startMessageName;   
}    
public String getStartMessageName() {
     return startMessageName;   
}    
//---------- Public Methods ------------------------------
// View commands
// Use edit value to set local value, display result
public void changeLocalValueCmd() {
    String newLocalValue = view.getEditValue();
    // Validate
    String problem = settingStore.getSettingDef(settingName)
        .validate(newLocalValue);
    if (problem != null) {
        GenLib.helpfulHint(
            "Invalid setting value entered.\n" + problem);
        view.requestFocusEditValue();
        return;
    }
    // Change local value
    settingStore.setLocalValue(settingName, newLocalValue);
    view.setLocalValue(newLocalValue);
}
// Get actual, set local, display result
public void revertToActualValueCmd() {
    String actual = settingStore.getActualValue(settingName);
    settingStore.setLocalValue(settingName, actual);
    view.setEditValue(actual);
    view.setLocalValue(actual);
}
public void closeCmd() {
    close();
}
//----- Other
public void runUnitTest() {
    print(" - Starting unit test");
    EditLocalSetting task = new EditLocalSetting();
    // Config like in _Container.parex
    task.setTitle("Unit test");
    task.setSettingName("Awards.CurrentFiscalYear");
    task.setFieldSize(6); // To hold year   
    // Done
    task.start();
}    
//---------- Private Methods -----------------------------
//--- Std
private static void print(String text) {
    System.out.println("EditLocalSetting" + text);
}

} // End class
