package org.jcon.util;

import java.util.Enumeration;

/**
* This interface is a set of key value pairs, aka properties.
* It's designed to make parameter driven designs easier to
* implement. It behavies similar to a Hashtable.
* <p>
* The key must be a String. Values can be any object,
* including null. Keys cannot be null.
*
* @author Jack Harich
*/
public interface BasicPropMap {

/**
* Sets a named property, overwriting any previous value
* that already exists with the same key.
*
* @param key  The unique key, which may not be null.
* @param value  the value, which may be null.
*
* @return the previous value or null if none.
*/
public Object set(String key, Object value);

/**
* Returns the named property or null if not found or
* the value is null.
*
* @param key  the unique key.
*
* @return the value for the key or null if none.
*/
public Object get(String key);

/**
* Removes the key and its value. This has no effect if the
* key is not in this collection. This method should be used 
* carefully since collaborators may be
* relying on this collection's structure.
*
* @param key  the desired key to remove.
*
* @return the value removed or null if none or value is null.
*/
public Object removeKey(String key);

/* REMOVED for now. Steve A.
* Same as removeKey(key) except it removes the first key
* with the argument value. This is a convenience method.
*
* @param key  the desired key to remove.
*
* @return the value removed or null if none or value is null.
*
public int removeValue(Object value);
*/

/**
* Used to determine if the set has the key.
* If get(key) returns null then containsKey(key) can be
* used to determine if this means the set has the
* key or the value is really null.
*
* @param key  the key in question.
*
* @return true if the set has the property, false if not.
*/
public boolean containsKey(String key);

/**
* Used to determine if the set has the value.
*
* @param key  the value in question.
*
* @return  true if the set has one or more references to
* the value, false if none.
*/
public boolean containsValue(Object value);


/**
* Returns an enumeration of all keys in the set.
*
* @return  the enumeration of keys.
*/ 
public Enumeration enumerateKeys();

/* REMOVED for now. Steve A.
* Returns an enumeration of all values in the set.
*
* @return  the enumeration of values, excluding nulls.
*
public Enumeration enumerateValues();
*/

/**
* Used to determine if the set is empty.
*
* @return  true if empty, false if contains one or more keys.
*/
public boolean isEmpty();

/*
* Used to determine how many keys are in the set.
*
* @return  the number of keys in the set, which is zero or more.
*/
public int getSize();

/**
* Replaces all of its own state with all state in the
* argument PropMap.
*
* @param  propMap  the propMap to copy state from.
*/
public void replaceAllWith(PropMap propMap);

/**
* Returns a comma delimited string of key/values, for
* example: "[Name=Pantajeli, Age=11]".
* This is designed to be easily readable.
*
* @return the String representation in standard format.
*/
public String toString();

/**
* Returns a copy of this instance. The actual values will 
* be the same instances of those values, so this is a 
* shallow clone.
* 
* @return a shallow clone of this instance.
*/
public Object clone();

}
