package org.jcon.ui.menu;

import org.jcon.util.msg.Message;
import org.jcon.util.msg.MessageListener;
import org.jcon.util.msg.MessageTranslator;
import org.jcon.ba.system.BasketServices;
import org.jcon.ba.system.BeanPriv;
import org.jcon.ba.system.BeanActionStart;
import org.jcon.param.Param;
import org.jcon.param.ParamAccessor;
import org.jcon.param.ParamStore;
import java.util.Vector;

/**
 * Manages a list of MenuItems, such as the ever popular
 * list of files last opened, or a list of open windows.
 * Note the extra work to make this reusable.
 *
 * @author Jack Harich
 */
public class MenuItemList implements
            BeanPriv, BeanActionStart, MessageListener {

//---------- Private Fields ------------------------------
private String menuName;
private Object markerClass;
private String paramFileName;
private String paramVectorName;
private String commandPrefix;

private Vector items = new Vector(); // String, itemName
private Vector menuItemListListeners = new Vector();
private MessageTranslator messageTranslator;
private BasketServices basketServices;

//---------- BeanPriv Implementation ---------------------
public void setBasketServices(BasketServices services) {
    basketServices = services;
}
//---------- BeanActionStart Implementation --------------
public void start() {
    loadList();
    sendList();
}
//---------- MessageListener Implementation --------------
public void processMessage(Message message) {
    if (messageTranslator != null) {
        message = messageTranslator.translate(message);
    }
    if (message.getName().equals("AddItem")) {
        addItemToList(message.getString("ItemName"));
    }
}
public String[] loadMessageInterests() {
    return new String[] {"AddItem"};
}
//---------- Properties ----------------------------------
//----- menuName
public void setMenuName(String menuName) {
    this.menuName = menuName;
}
public String getMenuName() {
    return menuName;
}
//----- markerClass
public void setMarkerClass(Object markerClass) {
    this.markerClass = markerClass;
}
public Object getMarkerClass() {
    return markerClass;
}
//----- paramFileName
public void setParamFileName(String paramFileName) {
    this.paramFileName = paramFileName;
}
public String getParamFileName() {
    return paramFileName;
}
//----- paramVectorName
public void setParamVectorName(String paramVectorName) {
    this.paramVectorName = paramVectorName;
}
public String getParamVectorName() {
    return paramVectorName;
}
//----- commandPrefix
public void setCommandPrefix(String commandPrefix) {
    this.commandPrefix = commandPrefix;
}
public String getCommandPrefix() {
    return commandPrefix;
}
//----- messageTranslator
public void setMessageTranslator(MessageTranslator translator) {
    messageTranslator = translator;
}
public MessageTranslator getMessageTranslator() {
    return messageTranslator;
}
//---------- Events --------------------------------------
public void addMenuItemListListener(MenuItemListListener listener) {
    menuItemListListeners.addElement(listener);
}
public void removeMenuItemListListener(MenuItemListListener listener) {
    menuItemListListeners.removeElement(listener);
}
//---------- Public Methods ------------------------------
public void addItemToList(String itemName) {
    //----- Maintain items
    // Remove if in items
    for (int i = 0; i < items.size(); i++) {
        String name = (String)items.elementAt(i);
        if (name.equals(itemName)) {
            items.removeElementAt(i);
            break;
        }
    }
    // Add to top of list
    items.insertElementAt(itemName, 0);
    // If over 9 items remove extra
    while (items.size() > 9) {
        items.removeElementAt(9);
    }
    sendList();
    saveList(); // No save if exception in sendList()
}
private void sendList() {
    //----- Fire event
    // Create MenuItemListEvent
    MenuItemListEvent event = new MenuItemListEvent();
    event.setMenuName(menuName);
    for (int i = 0; i < items.size(); i++) {
        String name = (String)items.elementAt(i);
        // Note the use of commandPrefix here
        event.add(name, commandPrefix + name);
    }
    // Send it
    Vector list;
    synchronized(this) {
        list = (Vector)menuItemListListeners.clone();
    }
    for (int i = 0; i < list.size(); i++) {
        MenuItemListListener listener =
            (MenuItemListListener)list.elementAt(i);
        listener.menuItemListChanged(event);
    }
}
//---------- Private Methods -----------------------------
private synchronized void loadList() {
    items = getParam().getVector(paramVectorName);
}
private synchronized void saveList() {
    // Modify the list of items
    Param param = getParam();
    if (param == null) return; // Error

    param.put(paramVectorName, items);
    basketServices.getParamStore().writeParam(param, null);
}
private Param getParam() {
    ParamAccessor accessor = new ParamAccessor();
    accessor.setMarkerClass(markerClass);
    accessor.setResourceName(paramFileName);
    return basketServices.getParamStore().readParam(accessor);
}
//--- Std
private static void print(String text) {
    System.out.println(text);
}

} // End class

/**
// MenuBar.parex - For the main menu menubar

TreeItems hasLines:
    org.jcon.ba.test1.Test
    End: TreeItems
*/
