package org.jcon.ui.easylayout;

import java.awt.*;

/**
 * Usage examples and unit test for EasyLayout. Note the
 * frequent use of styles, marked with a <=====.
 * Test: java org.jcon.ui.easylayout.EasyLayoutExamples
 *
 * @author Jack Harich
 */
public class EasyLayoutExamples {

//---------- Private Fields ------------------------------
private EasyLayout layout = new EasyLayout();
private int        count, frameX, frameY;

//---------- Initialization ------------------------------
public EasyLayoutExamples() {

    // Create and store GapTopBottom style
    layout.useStyle("Default"); // <=====
    layout.setInsetTop(3);
    layout.setInsetBottom(3);
    layout.storeCurrentStyle("GapTopBottom"); // <=====

    // Use default insets of all zeros - too close
    doFieldsExample("Too Close");

    // Mimic BorderLayout a bit
    doStatusBarExample();
    doStatusBarExampleWithLibrary();

    // Several others
    doPartitionedExample();
    doBorderLayout();
    doBorderLayoutWithRegistry();
    doSideButtons();
    // Minor doTwoButtons();
}
public static void main(String args[]) {
    new EasyLayoutExamples();
}
//---------- Public Methods ------------------------------
//----- Examples
// Components stay centered when stretched
public void doFieldsExample(String title) {
    Frame frame = createFrame(title);
    layout.setTopContainer(frame);
    // *** remove layout.useStyle("GapTopBottom"); // <===== ***
    addStandardRows();
    showFrame(frame);
}
/**
 * Uses styles for a medium complexity layout.
 */
 // Components stay left aligned when stretched
public void doStatusBarExample() {
    Frame frame = createFrame("Status Bar, Several Styles Used");
    layout.setTopContainer(frame);

    // Create TopBar style
    layout.useStyle("Default"); // <=====
    layout.setInsetTop(2);
    layout.setInsetRight(1);
    layout.setStretch("Width");
    layout.setWidthWeight(1);
    layout.setHeightWeight(1); // To allow to float up
    layout.setAlignment("Top");
    layout.storeCurrentStyle("TopBar"); // <=====

    // Add status bar
    layout.add(createStatusBar(), "TopBar", "EndRow");


    /** layout.useStyle("Default");
    
    layout.setAlignment("Right");
    layout.add(new Label("TEST"));
    
    layout.setAlignment("Left");
    layout.add(new TextField(10));
    
    // *** Causes fields to be left aligned
    // *** Without this "filler" GridBagLayout apparently
    //     clumps the fields in the middle
    layout.setWidthWeight(1);
    layout.addEndRow(new Label()); */


    // Field rows
    layout.useStyle("GapTopBottom"); // <=====
    addStandardRows();

    // Button bar
    layout.useStyle("TopBar"); // <=====
    layout.setAlignment("Bottom");
    layout.storeCurrentStyle("BottomBar"); // <=====
    layout.add(createButtonBar(), "BottomBar", "EndRow");

    showFrame(frame);
}
/**
 * Uses library styles for a medium complexity layout.
 */
public void doStatusBarExampleWithLibrary() {
    Frame frame = createFrame("Status Bar - Using Library");
    layout.setTopContainer(frame);

    // Use standard library styles in layout
    EasyLayoutLibrary library = new EasyLayoutLibraryStd();
    // the old way - library.storeAllStyles(layout);
    layout.setLibrary(library); // New improved way

    // Add status bar <=====
    layout.add(createStatusBar(), "TopBar", "EndRow");

    // Field rows <=====
    layout.useStyle("GapTopBottom");
    addStandardRows();

    // Button bar <=====
    layout.add(createButtonBar(), "BottomBar", "EndRow");

    showFrame(frame);
}
private TextField createStatusBar() {
    TextField statusBar = new TextField();
    statusBar.setBackground(Color.lightGray);
    statusBar.setText("This sure is easy to use...");
    statusBar.setEditable(false);
    return statusBar;
}
public void doPartitionedExample() {
    Frame frame = createFrame("Stretch Me, Please");
    layout.setTopContainer(frame);
    layout.useStyle("Default"); // <=====
    // Left list
    layout.setRowColumn(1, 1);
    layout.setGridWidthHeight(1, 2);
    layout.setStretch("Height");
    layout.setHeightWeight(1);
    layout.add(new List(10));
    // Right upper list
    layout.setRowColumn(1, 2);
    layout.setGridWidthHeight(1, 1);
    layout.setWidthWeight(1);
    layout.setStretch("Both");
    layout.add(new List());
    // Right lower list
    layout.setRowColumn(2, 2);
    layout.setStretch("Both");
    layout.add(new List());
    // Bottom button bar
    layout.useStyle("BottomBar");
    layout.setHeightWeight(0);
    layout.setRowColumn(3, 1);
    layout.add(createButtonBar(), null, "EndRow");

    showFrame(frame);
}
/**
 * Demonstrates use of nested containers. Note the use of
 * indentation and the high readability compared to using
 * GridBagLayout directly.
 */
public void doSideButtons() {
    Frame frame = createFrame("Side Buttons");
    layout.useStyle("GapTopBottom"); // <=====
    layout.setTopContainer(frame);

    // Central fields
    layout.setRowColumn(1, 1);
    layout.add(new Panel());
    layout.beginNestedContainer();
        layout.useStyle("Default"); // <=====
        layout.setInsets(6, 4, 6, 4); // Separate a bit more
        layout.add(new Label("Please enter the following"
            + " information:"), null, "EndRow");
        addStandardRows();
        layout.endNestedContainer();
    // Right vertical button column
    layout.setRowColumn(1, 2);
    layout.add(new Panel());
    layout.beginNestedContainer();
        layout.storeStyle("SingleColumn", createSingleColumnStyle());
        layout.useStyle("SingleColumn"); // <=====
        layout.add(new Button("Commit"));
        layout.add(new Button("Forward"));
        layout.add(new Panel());
        layout.beginNestedContainer();
            layout.useStyle("Default"); // <=====
            layout.add(new Button("A"));
            layout.add(new Button("B"));
            layout.add(new Button("C"));
            layout.endNestedContainer();
        layout.add(new Button("Snarf"));
        layout.add(new Button("Go To Lunch"));
        layout.add(new Button("Cancel"));
        layout.add(new Label(""));
        layout.setExtraSizeHeight(10);
        layout.add(new Button("Close"));
        layout.endNestedContainer();
    // Bottom status bar
    layout.useStyle("BottomBar"); // <=====
    layout.setRowColumn(2, 1);
    layout.addEndRow(createStatusBar());

    showFrame(frame);
}
/**
* Demonstrates ability to have 2 components "equally"
* centered in a container.
*/
private void doTwoButtons() {
    Frame frame = createFrame("Equally Centered");
    layout.useStyle("Default"); // <=====
    layout.setTopContainer(frame);
    
    layout.setHeightWeight(1);
    layout.setWidthWeight(1);
    layout.addEndRow(new Button("First"));
    layout.addEndRow(new Button("Second"));
    
    showFrame(frame);
    frame.setSize(250, 200);
}
private EasyLayoutStyle createSingleColumnStyle() {
    EasyLayoutStyle style = new EasyLayoutStyle("SingleColumn");
    style.setInsets(4, 4, 4, 4);
    style.setGridWidth(0); // For end of row
    style.setStretch("Width"); // All same width
    return style;
}
/**
 * Note the ease of creating a reusable custom layout.
 * This is what "reusables designed for configuration"
 * is all about.
 */
public void doBorderLayout() {
    //----- Create styles // <===== many

    // North, South
    layout.useStyle("Default");
    layout.setRowColumn(1, 1);
    layout.setGridWidthHeight(3, 1);
    layout.setStretch("Width");
    layout.storeCurrentStyle("BorderNorth");

    layout.setRowColumn(3, 1);
    layout.storeCurrentStyle("BorderSouth");

    // East, West
    layout.useStyle("Default");
    layout.setRowColumn(2, 3);
    layout.setHeightWeight(1);
    layout.setStretch("Height");
    layout.storeCurrentStyle("BorderEast");

    layout.setRowColumn(2, 1);
    layout.storeCurrentStyle("BorderWest");

    // Center
    layout.useStyle("Default");
    layout.setRowColumn(2, 2);
    layout.setHeightWeight(1);
    layout.setWidthWeight(1);
    layout.setStretch("Both");
    layout.storeCurrentStyle("BorderCenter");

    //----- Use styles in concise manner
    Frame frame = createFrame("Border Layout - Stretch Me");
    layout.setTopContainer(frame);
    // Add in desired tab order
    layout.add(new Button("North"),  "BorderNorth");
    layout.add(new Button("West"),   "BorderWest");
    layout.add(new Button("Center"), "BorderCenter");
    layout.add(new Button("East"),   "BorderEast");
    layout.add(new Button("South"),  "BorderSouth");

    showFrame(frame);
    frame.setSize(400, 300);
}
/**
 * A BorderLayout using the registry and library.
 */
public void doBorderLayoutWithRegistry() {
    String LIBRARY_NAME = "org.jcon.Standard";

    // Register library
    // Generally done in system or subsystem initialization
    EasyLayoutLibraryStd library = new EasyLayoutLibraryStd();
    EasyLayoutRegistry.addLibrary(LIBRARY_NAME, library);

    // Create a layout using the library
    EasyLayout localLayout = EasyLayoutRegistry
        .createLayout(LIBRARY_NAME);

    Frame frame = createFrame("Border Layout - Using Registry");
    localLayout.setTopContainer(frame);
    // Add in desired tab order
    localLayout.add(new Button("Top"),    "BorderTop");
    localLayout.add(new Button("Left"),   "BorderLeft");
    localLayout.add(new Button("Center"), "BorderCenter");
    localLayout.add(new Button("Right"),  "BorderRight");
    localLayout.add(new Button("Bottom"), "BorderBottom");
    showFrame(frame);
    frame.setSize(400, 300);
}
//----- Helpers
public void addStandardRows() {
    // Add rows, a Label and TextField on each
    layout.setInsetTop(layout.getInsets().top + 3);
        addFieldsRow("Full Name", 25);
    layout.setInsetTop(layout.getInsets().top - 3);
    addFieldsRow("Rank", 10);
    addFieldsRow("Address", 20);

    // An example of a field with no label to demonstrate addNothing()
    // Could also use addFieldsRow("", 20);
    layout.addNothing();
    layout.setAlignment("Left");
    layout.addEndRow(new TextField(20));

    addFieldsRow("Serial Number", 15);
}
public void addFieldsRow(String text, int fieldColumns) {
    layout.setAlignment("Right");
    layout.add(new Label(text));

    layout.setAlignment("Left");
    layout.add(new TextField(fieldColumns), null, "EndRow");
}
public Panel createButtonBar() {
    Panel buttonBar = new Panel();
    buttonBar.add(new Button("Add"));
    buttonBar.add(new Button("Change"));
    buttonBar.add(new Button("Delete"));
    buttonBar.add(new Button("Close"));
    return buttonBar;
}
//---------- Private Methods -----------------------------
private Frame createFrame(String title) {
    Frame frame = new Frame(" " + ++count + " - " + title);
    frame.setBackground(Color.lightGray);
    frame.setLocation(frameX, frameY);
    frameX += 30;
    frameY += 30;
    return frame;
}
private void showFrame(Frame frame) {
    frame.pack();
    frame.setVisible(true);
}
//--- Std
private static void print(String text) {
    System.out.println("EasyLayoutExamples" + text);
}

} // End class
