package org.jcon.ui;

import org.jcon.util.GenLib;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Toolkit;
import java.io.InputStream;
import javax.swing.ImageIcon;
import javax.swing.JPanel;

/**
 * Provides an image inside of a JPanel, blocking until
 * the image is loaded. Note that a modification could
 * be made to provide optional asynch image loading.
 *
 * The image may be provided or loaded as a resource of a
 * class, which is the usual technique. For example:
 * JImageCanvas canvas = new JImageCanvas(this, "Snazzy.gif");
 *
 * The canvas is sized to fit the image exactly.
 *
 * @author Jack Harich, Cindy Conway
 */
public class JImageCanvas extends JPanel {

//---------- Private Fields ------------------------------
private ImageIcon image;
private String resourceClassName;
private String resourceName;

//---------- Initialization ------------------------------
/**
 * Creates a Canvas containing the Image.
 */
public JImageCanvas(Image inImage) {
    image = new ImageIcon(inImage);
}
/**
 * Creates a Canvas containing an Image loaded from the
 * resourceName of resourceClass.
 */
public JImageCanvas(Class resourceClass, String resourceName) {
    initWithClass(resourceClass, resourceName);
} 

/**
 * Creates a Canvas containing an Image loaded from the
 * resourceName of resourceClass.
 */
public JImageCanvas(String resourceClass, String resourceName) {
    try {
        Class theClass = Class.forName(resourceClass);
        initWithClass(theClass, resourceName);
    } catch(Exception ex) {
        GenLib.exception("JImageCanvas()", "Cannot load '" 
				+ resourceName + "' with class '" 
				+ resourceClass + "'.", ex);    
    }
}
//---------- Superclass Overrides ------------------------
public void paint(Graphics g) {
    g.drawImage(image.getImage(), 0, 0, this);
}
public Dimension getPreferredSize() {
    //print(".getPreferredSize() - " 
	//        + image.getImage().getWidth(this) 
	//        + ". " +  image.getImage().getHeight(this));
    return new Dimension(image.getImage().getWidth(this), 
					image.getImage().getHeight(this));
}
//---------- Properties ----------------------------------
public Image getImage() {
    return image.getImage();
}
//---------- Private Methods -----------------------------
private void initWithClass(Class resourceClass, String resourceName) {
    resourceClassName = resourceClass.getName();
    this.resourceName = resourceName;
    try {
        InputStream resource =
            resourceClass.getResourceAsStream(resourceName);
        byte[] bytes = new byte[resource.available()];
        resource.read(bytes);
		Image img = Toolkit.getDefaultToolkit().createImage(bytes);
        image = new ImageIcon(img);
    
    } catch(Exception ex) {
        GenLib.exception("JImageCanvas.initWithClass()", "Cannot load '" 
			+ resourceName + "' with class '" 
			+ resourceClass.getName() + "'.", ex);
    }
}

//--- Std
private static void print(String text) {
    System.out.println("JImageCanvas" + text);
}

} // End Class
