package org.jcon.ui;

import org.jcon.util.msg.Message;
import org.jcon.util.msg.MessageListener;
import java.awt.*;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;

/**
 * Displays information, such as text or icons, in a status
 * bar style window. This is located at the bottom of the
 * screen, and hides when the app does not have the focus.
 * <p> <pre>
 * Potential future features are:
 * - Automatic size to fit number of lines of text.
 * - Dockable to top or bottom of screen
 * - Dynamically add/remove other components, such as
 *      a task status light with green for normal, yellow
 *      for caution, red for stop since busy. </pre>
 *
 * @author Jack Harich
 */
public class InfoBar implements MessageListener {

//---------- Private Fields ------------------------------
private Window    window;
private Frame     dummyFrame;
private TextField textField = new TextField();
private boolean   initialized;

//---------- Initialization ------------------------------
private void init() {
    // Config textField
    textField.setBackground(Color.lightGray);
    textField.setEditable(false);

    // Config window
    dummyFrame = new Frame("Dummy");
    window = new Window(dummyFrame);
    // Failed *** window.setEnabled(false); // To solve focus problem
    window.add("North", textField);
    window.pack();

    // Get screen and window dimensions
    Dimension screenDim = Toolkit.getDefaultToolkit()
        .getScreenSize();
    Dimension windowDim = window.getSize();

    // Size window
    window.setSize(screenDim.width, windowDim.height);

    // Position window at bottom of screen
    window.setLocation(0, screenDim.height - windowDim.height);
}

public static void main(String args[]) {
    new InfoBar().runUnitTest();
}
//---------- Superclass Overrides ------------------------

//---------- MessageListener Implementation --------------
public void processMessage(Message message) {
    String messageName = message.getName();

    if (messageName == "SetStatusBarText") {
        setText(message.getString("Text"));

    } else if(messageName == "SetInfoBarVisible") {
        setVisible(message.isTrue("IsVisible"));

    } else if(messageName == "SystemActivated") {
        // *** Bug - This causes the active window to lose
        // focus, which if the only other window causes a
        // SystemDeactivated event.
        // See gov.cdc.atsdr.client.Mediator
        window.toFront();

    } else if(messageName == "SystemDeactivated") {
        window.toBack();

    } else {
        print(".processMessage() - Unknown messageName '" + messageName + "'");
    }
}
public String[] loadMessageInterests() {
    return new String[] {"SetStatusBarText",
                         "SetStatusBarVisible",
                         "SystemActivated",
                         "SystemDeactivated" };
}
//---------- Properties ----------------------------------
/**
 * Displays the provided text.
 */
public void setText(String text) {
    textField.setText(text);
}
/**
 * Sets the window to visible or not visible.
 */
public void setVisible(boolean visible) {
    if (visible && ! initialized) {
        init();
        initialized = true;
    }
    window.setVisible(visible);
}
/**
 * Destroys the window and related resources, returning the
 * instance to its post constructor state. It may then be
 * reused.
 */
public void close() {
    window.setVisible(false);
    window.dispose();
    window = null;

    dummyFrame.dispose();
    dummyFrame = null;

    setText("");
    initialized = false;
}
//---------- Public Methods ------------------------------
public void runUnitTest() {
    InfoBar bar = new InfoBar();
    bar.setText("This contains useful descriptive text.");
    bar.setVisible(true);
}
//---------- Private Methods -----------------------------
//--- Std
private static void print(String text) {
    System.out.println("InfoBar" + text);
}

} // End class
