package org.jcon.ui;

import org.jcon.util.GenLib;
import java.awt.*;
import java.io.InputStream;

/**
 * Provides an image inside of a canvas, blocking until
 * the image is loaded. Note that a modification could
 * be made to provide optional asynch image loading.
 *
 * The image may be provided or loaded as a resource of a
 * class, which is the usual technique. For example:
 * ImageCanvas canvas = new ImageCanvas(this, "Snazzy.gif");
 *
 * The canvas is sized to fit the image exactly.
 *
 * @author Jack Harich
 */
public class ImageCanvas extends Canvas {

//---------- Private Fields ------------------------------
private Image image;
private String resourceClassName;
private String resourceName;

//---------- Initialization ------------------------------
/**
 * Creates a Canvas containing the Image.
 */
public ImageCanvas(Image image) {
    this.image = image;
    loadImage();
}
/**
 * Creates a Canvas containing an Image loaded from the
 * resourceName of resourceClass.
 */
public ImageCanvas(Class resourceClass, String resourceName) {
    initWithClass(resourceClass, resourceName);
} 

/**
 * Creates a Canvas containing an Image loaded from the
 * resourceName of resourceClass.
 */
public ImageCanvas(String resourceClass, String resourceName) {
    try {
        Class theClass = Class.forName(resourceClass);
        initWithClass(theClass, resourceName);
    } catch(Exception ex) {
        GenLib.exception("ImageCanvas()", "Cannot load '" + resourceName + "' with class '" + resourceClass + "'.", ex);    
    }
}
//---------- Superclass Overrides ------------------------
public void paint(Graphics g) {
    g.drawImage(image, 0, 0, this);
}
public Dimension getPreferredSize() {
    //print(".getPreferredSize() - " + image.getWidth(this) + ". " +  image.getHeight(this));
    return new Dimension(image.getWidth(this), image.getHeight(this));
}
//---------- Properties ----------------------------------
public Image getImage() {
    return image;
}
//---------- Private Methods -----------------------------
private void initWithClass(Class resourceClass, String resourceName) {
    resourceClassName = resourceClass.getName();
    this.resourceName = resourceName;
    try {
        InputStream resource =
            resourceClass.getResourceAsStream(resourceName);
        byte[] bytes = new byte[resource.available()];
        resource.read(bytes);
        image = Toolkit.getDefaultToolkit().createImage(bytes);
        loadImage();
    
    } catch(Exception ex) {
        GenLib.exception("ImageCanvas.initWithClass()", "Cannot load '" + resourceName + "' with class '" + resourceClass.getName() + "'.", ex);
    }
}
private void loadImage() {

    MediaTracker tracker = new MediaTracker(this);
    tracker.addImage(image, 0);

    try {
        tracker.waitForID(0);

    } catch(InterruptedException e){
        GenLib.error("ImageCanvas.loadImage()",
            "Cannot load '" + resourceName + "' with class '" + resourceClassName + "'.");
    }
    setSize(image.getWidth(null), image.getHeight(null));
}
//--- Std
private static void print(String text) {
    System.out.println("ImageCanvas" + text);
}

} // End Class
