package org.jcon.ui;

import java.awt.*;
import javax.swing.JLabel;

/**
 * Various library methods useful with GridBagLayout.
 * These are heavily used by EditorView and it's subclasses.
 *
 * All methods are static.
 *
 * The container in the method arguments must use
 * GridBagLayout.
 *
 * @author Jack Harich
 */

public class GridBagLib {

//---------- Private Fields ------------------------------
private static final int REMAINDER  = GridBagConstraints.REMAINDER;
private static final int HORIZONTAL = GridBagConstraints.HORIZONTAL;

private static final int NONE = GridBagConstraints.NONE;
private static final int EAST = GridBagConstraints.EAST;
private static final int WEST = GridBagConstraints.WEST;

//---------- Public Methods ------------------------------
/**
 * Adds a Label and Component row to the Container. The
 * label is added to the first column and the Component
 * is added to the remainder of columns, which causes
 * a new row to start.
 */
public static void addLabeledComponent(String labelText,
        Component component, Container container) {

    GridBagConstraints constraints = getDefaultConstraints();
    GridBagLayout gbLayout = (GridBagLayout)container.getLayout();

    // Put label in 1st column
    Label label = new Label(labelText);
        // Insets() args are top, left, bottom, right
    constraints.insets   = new Insets(4,4,4,0);
    constraints.gridwidth = 1; // One column wide
    constraints.anchor = EAST;
    gbLayout.setConstraints(label, constraints);
    container.add(label);

    // Put Component in remaining columns
        // Insets() args are top, left, bottom, right
    constraints.insets   = new Insets(4,0,4,4);
    constraints.gridwidth = REMAINDER; // Last on right
    constraints.anchor = WEST;
    gbLayout.setConstraints(component, constraints);
    container.add(component);
}
/**
 * Provides a consistent GridBagConstraints useful as a
 * starting point in layout code. Initial values are
 * weightx = 1.0, weighty = 1.0, gridheight = 1,
 * insets = (4, 4, 4, 4), fill = NONE, anchor = WEST.
 */
public static GridBagConstraints getDefaultConstraints() {
    GridBagConstraints constraints = new GridBagConstraints();
    constraints.weightx  = 1.0;
    constraints.weighty  = 1.0;
    constraints.gridheight = 1; // One row high
        // Insets() args are top, left, bottom, right
    constraints.insets   = new Insets(4,4,4,4);
        // fill of NONE means do not change size
    constraints.fill = NONE;
        // WEST means align left
    constraints.anchor = WEST;

    return constraints;
}
/**
 * Adds a blank row @param pixelHeight high to the
 * Container. Currently done with an invisible Canvas.
 */
public static void addSeparator(int pixelHeight,
        Container container) {
    GridBagConstraints constraints = getDefaultConstraints();
    GridBagLayout gbLayout = (GridBagLayout)container.getLayout();

    //--- Use Canvas for spacer. Label failed to work.
    Canvas canSpacer = new Canvas();
//canSpacer.setBackground(Color.red);
    constraints.gridwidth = REMAINDER; // Whole row
    gbLayout.setConstraints(canSpacer, constraints);
    container.add(canSpacer);
    canSpacer.setSize(5, pixelHeight); // width irrelevant
}
/**
 * Adds one row per line in message to the Container.
 * Lines are delimited with the usual \n.
 *
 * This method is useful for message boxes.
 *
 * NOTE - Currently the text lines are a little far apart
 * due to using one Label per line.
 * This should be reduced in the future. **********
 */
public static void addMultilineString(String message,
        Container container) {

    GridBagConstraints constraints = getDefaultConstraints();
    constraints.gridwidth = REMAINDER;
        // Insets() args are top, left, bottom, right
    constraints.insets   = new Insets(0,6,0,0);
    GridBagLayout gbLayout = (GridBagLayout)container.getLayout();

    while (message.length() > 0) {
        int newLineIndex = message.indexOf('\n');
        String line;
        if (newLineIndex >= 0) {
            line = message.substring(0, newLineIndex);
            message = message.substring(newLineIndex + 1);
        } else {
            line = message;
            message = "";
        }
        JLabel label = new JLabel(line);
        gbLayout.setConstraints(label, constraints);
        container.add(label);
    }
}
//---------- Private Methods -----------------------------
// (none)

} // End class
