package org.jcon.ui;

import org.jcon.ui.VisualLib;
import org.jcon.ui.WindowLib;
import java.awt.Color;
import java.awt.BorderLayout;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import javax.swing.JFrame;

/**
 * Presents a "delay window" for use while a process is
 * slow to start. Currently this is a frame for simplicity.
 * See main() for unit test and use example.
 * Test: java org.jcon.ui.DelayWindow
 *
 * @author Jack Harich
 */
public class DelayWindow implements
    Runnable, WindowListener {

//---------- Private Fields ------------------------------
private JFrame    window = new JFrame(); // Not registered
private ColorBar colorBar = new ColorBar();
private boolean  enforceMinSize;
private int      currentCell = 0;
private boolean  running;
private int      delayUntilAutoShow = 0;

//---------- Initialization ------------------------------
public DelayWindow() {
    window.setBackground(Color.lightGray);
    window.getContentPane().setLayout(new BorderLayout(5, 5));
    window.addWindowListener(this);

    colorBar.setCellCount(15);
    window.getContentPane().add("South", colorBar);
}
public static void main(String args[]) { // Unit test
    DelayWindow delay = new DelayWindow();
    delay.setTitle("Patience Is A Virtue");
    delay.setMessage(
        "Constructing task from parameters\n" +
        "and loading data from server.");
    delay.start();
}
//---------- Runnable Implementation ---------------------
/**
 * For internal use only. Clients should not call this method.
 */
public void run() {
    try {
        // delayUntilAutoShow feature
        Thread.currentThread().sleep(delayUntilAutoShow);
        if (! running) return;
        window.setVisible(true);

        while (running) {
            // Advance one cell per second
            Thread.currentThread().sleep(1000);
            if (! running) return;

            currentCell++;
            if (currentCell > colorBar.getCellCount()) currentCell = 0;
            colorBar.setFullCount(currentCell);
            colorBar.repaintNow();

			paintWindow();

            //print(".run() - Advanced one cell");
        }
    } catch(InterruptedException ex) {
        // Ignore
    }
}
//---------- WindowListener Implementatons ---------------
public void windowClosing(WindowEvent evt) {
    // User clicked on X or chose window Close
    stop();
}
public void windowClosed(WindowEvent evt) { }
public void windowDeiconified(WindowEvent evt) { }
public void windowIconified(WindowEvent evt) { }
public void windowOpened(WindowEvent evt) { }
public void windowActivated(WindowEvent evt) {  }
public void windowDeactivated(WindowEvent evt) { }

//---------- Properties ----------------------------------
/**
 * Sets the window title.
 */
public void setTitle(String title) {
    window.setTitle(title);
}
/**
 * Sets the message to be displayed, which may be multi-line.
 */
public void setMessage(String message) {
    window.getContentPane().add("Center", VisualLib.createMultiLinePanel(message));
}
/**
 * Sets the EnforceNormalMinimumSize property. If true then
 * the window will be at least as big as the normal minimum
 * window size, which is width 200, height 100.
 * The default is false.
 */
public void setEnforceNormalMinimumSize(boolean enforceMinimumSize) {
    enforceMinSize = enforceMinimumSize;
}
/**
 * Sets the color bar cell count. The default is 15.
 */
public void setCellCount(int cellCount) {
    colorBar.setCellCount(cellCount);
}
/**
 * Sets the DelayUntilAutoShow property in milliseconds.
 * The default is 0, which will delay the show of the
 * window for 0 miliseconds after start is called.
 * A value of zero will cause the window to show immediately
 * after start() is called.
 * <p>
 * If stop() is called before the DelayUntilAutoShow expires
 * then the window is never shown.
 */
public void setDelayUntilAutoShow(int delayUntilAutoShow) {
    this.delayUntilAutoShow = delayUntilAutoShow;
}
//---------- Public Methods ------------------------------
/**
 * Starts the window by using the supplied properties to
 * configure it, and then showing it. Since the window is
 * reused, properties are reused also, and thus may be set
 * only once. However minimum size is enforced each time
 * and the window is centered each time.
 */
public void start() {
    window.pack();
    if (enforceMinSize) {
        WindowLib.enforceMinimumSize(window, 200, 100);
    }
    WindowLib.center(window);

    // Start cell advancer
    running = true;
    Thread thread = new Thread(this);
    thread.setPriority(Thread.NORM_PRIORITY + 2);
    new Thread(this).start();
    //print(".start() - Leaving");
}
/**
* Same as start() except it first sets the DelayUntilAutoShow.
* This is the same as calling setDelayUntilAutoShow(delay)
* and then start().
*/
public void start(int delayUntilAutoShow) {
    setDelayUntilAutoShow(delayUntilAutoShow);
    start();
}
/**
 * Stops the window by hiding it.
 */
public void stop() {
    window.setVisible(false);
    running = false;
}
/**
 * Closes the window by stopping and disposing of it. This
 * should be called when the client is done with the instance.
 */
public void close() {
    stop();
    window.dispose();
}
//---------- Private Methods -----------------------------
//--- Std
private static void print(String text) {
    System.out.println("DelayWindow" + text);
}

/**
 * Repaints all components on the window 
 * @author C. Conway
 */
public void paintWindow() {
	window.paintComponents(window.getGraphics());
}

} // End class
