package org.jcon.df.work;

import java.util.Enumeration;

/**
 * Fired when a RowSet state change has occurred. The wide
 * variety of event types allows system simplification
 * since a single event is used. A small number of events
 * are "acquire" events, used to acquire data.
 *
 * @author Jack Harich
 */
public class RowSetEvent {

//---------- Public Fields -------------------------------
public static final int CURRENT_ROW_INDEX_CHANGED = 1;
public static final int CURRENT_ROW_DATA_CHANGED  = 2;
public static final int MOVE_STATE_CHANGED = 3;
public static final int EDIT_STATE_CHANGED = 4;
public static final int ACQUIRE_EDIT_ROW   = 5;
public static final int REQUEST_FOCUS      = 6;
public static final int ALL_ROWS_CHANGED   = 7;
public static final int ROWS_INSERTED      = 8;
public static final int ROWS_DELETED       = 9;
public static final int CLEAR_ROW          = 10;

//---------- Private Fields ------------------------------
private int    eventType;
private RowSet rowSet;
private Row    editRow;

private int    firstIndexAffected;
private int    lastIndexAffected;

//---------- Initialization ------------------------------
public RowSetEvent(int eventType, RowSet rowSet) {
    this.eventType = eventType;
    this.rowSet = rowSet;
}
//---------- Superclass Overrides ------------------------
public String toString() {
    return getEventTypeString();
}
//---------- Properties ----------------------------------
//----- eventType
public String getEventTypeString() {
    if (eventType == CURRENT_ROW_INDEX_CHANGED) return "CurrentRowIndexChanged";
    if (eventType == CURRENT_ROW_DATA_CHANGED) return "CurrentRowDataChanged";
    if (eventType == MOVE_STATE_CHANGED) return "MoveStateChanged";
    if (eventType == EDIT_STATE_CHANGED) return "EditStateChanged";
    if (eventType == ACQUIRE_EDIT_ROW) return "AcquireEditRow";
    if (eventType == REQUEST_FOCUS) return "RequestFocus";
    if (eventType == ALL_ROWS_CHANGED) return "AllRowsChanged";
    if (eventType == ROWS_INSERTED) return "RowsInserted";
    if (eventType == ROWS_DELETED) return "RowsDeleted";
    if (eventType == CLEAR_ROW) return "ClearRow";

    return "#UnknownType#";
}
public int getEventType() {
    return eventType;
}
//----- editRow
/**
* Sets the EditRow to the row. This is useful for a single
* source of the row. Multiple sources should use 
* addRowProperties(Row), such as RowFieldPanel since a
* view may have multiple RowFieldPanels with the same
* source.
*/
public void setEditRow(Row row) {
    editRow = row;
}
/**
* Adds the row's properties to the EditRow. See
* setEditRow(Row) for further documentation.
*/
public void addRowProperties(Row row) {
    if (editRow == null) editRow = new Row();
    Enumeration keys = row.getColumnIDs();
    while(keys.hasMoreElements()) {
        String key = (String)keys.nextElement();
        Object value = row.getValue(key);
        editRow.setValue(key, value);
    }
}
public Row getEditRow() {
    return editRow;
}
//----- firstIndexAffected
public void setFirstIndexAffected(int index) {
    firstIndexAffected = index;
}
public int getFirstIndexAffected() {
    return firstIndexAffected;
}
//----- lastIndexAffected
public void setLastIndexAffected(int index) {
    lastIndexAffected = index;
}
public int getLastIndexAffected() {
    return lastIndexAffected;
}
//----- Other
public Row getCurrentRow() {
    return rowSet.getCurrentRow();
}
public String getMoveState() {
    return rowSet.getMoveState();
}
public String getEditState() {
    return rowSet.getEditState();
}
public int getRowCount() {
    return rowSet.getRowCount();
}
public Row getRowAt(int index) {
    return rowSet.getRowAt(index);
}
public int getCurrentIndex() {
    return rowSet.getCurrentIndex();
}
public String getFocusColumnID() {
    return rowSet.getFocusColumnID();;
}
public RowSetReader getRowSetReader() {
    return rowSet.getRowSetReader();
}
//---------- Private Methods -----------------------------
//--- Std
private static void print(String text) {
    System.out.println("RowSetEvent" + text);
}

} // End class
