package org.jcon.df.work;

import org.jcon.util.PropMapStd;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

/**
* This class carries a collection of key value pairs, aka
* columns. Each column has a columnID and value. If a row
* is an entity row then columnID = "EntityName.ColumnName".
* Otherwise columnID can be any indentifier.
* <p>
* Null values are fully supported. The value is usually a
* String. Object values can be used for anything, such as
* drilling down, images, etc.
* <p>
* Many convenience methods are supplied so that gets and
* sets are concise, regardless of the value type.
*
* @author Jack Harich
*/

// 10/12/98 - Modified to extend PropMap, various methods
//            removed or rewritten.
// 5 July 1999 Changed to extend PropMapStd: Steve A.

public class Row extends PropMapStd {

//---------- Superclass Overrides ------------------------
/**
* Returns a copy of this instance. The actual values will 
* be the same instances of those values, so this is a 
* shallow clone.
* 
* @return a shallow clone of this instance.
*/
public synchronized Object clone() {
    Row row = new Row();
    row.properties = (Hashtable)properties.clone();
    return row;
}
//---------- Properties ----------------------------------
/**
* Returns an Enumeration of all columnIDs in this row, in
* no particular order. 
*
* @return an Enumeration of Strings. Each is a columnID.
*/
public Enumeration getColumnIDs() {
    return properties.keys();
}
//---------- Public Methods ------------------------------
/**
* Returns the value for the column named columnID.
* If the column doesn't exist an IllegalArgumentException is
* thrown. If the value is null then null is returned. If not
* found an IllegalArgumentException is thrown, since this
* is usually a bug.
*
* @param  columnID  The unique identifier for a key value.
*
* @return the value associated with the columnID, aka key.
*/
public Object getValue(String columnID) {
    if (containsKey(columnID)) {
      return get(columnID);
    } else {
        throw new IllegalArgumentException(
            "Cannot find columnID '" + columnID +
            "' in row:\n" + this);
    }
}
/**
* Sets the value for the column named columnID.
* If the column doesn't exist it is added. 
*
* @param  columnID  the unique identifier for the value.
* @param  value     the value to store. May be null.
*/
public void setValue(String columnID, Object value) {
    super.set(columnID, value);
}
/**
* Copies all state in the row to itself, completely 
* replacing all previous internal state. 
*
* @param  row  the row to copy state from.
*/
public void copy(Row row) {
    super.replaceAllWith(row);
}
/**
* Returns true if this row contains the columnID, false if
* not. Use this method if you are not sure if getValue()
* can be called safely.
*
* @param  columnID  the columnID in question.
* @return           true if has a value for the columnID,
*                     false otherwise.
*/
public boolean hasColumnID(String columnID) {
    return properties.containsKey(columnID);
}
//----- Convenience Methods
/**
* Returns true if the value is "1" or true, false if is
* "0" or false, and otherwise throws an IllegalStateException.
* This supports legacy code in the early Data Framework,
* and should be redesigned and removed. - JH 10/12/98
*/
public boolean isTrue(String columnID) {
    Object value = getValue(columnID);
    if (value.equals("1")) {
        return true;
           
    } else if (value.equals("0")) {
        return false;

    } else {
        return super.isTrue(columnID);
        //throw new IllegalStateException("Boolean value "
        //    + "must be 1 or 0, true or false, not '" + value + "'.");
    }
}  
/**
* Returns the int for the value, which may be a String or
* an Integer. The String feature supports legacy code and
* should be redesigned and removed. - JH 10/12/98
*/
public int getInt(String columnID) {
    Object value = super.get(columnID);
    if (value instanceof String) {
        return Integer.parseInt((String)value);
    } else {
        return ((Integer)value).intValue();
    }
}
/**
* Returns the long for the value, which may be a String or
* a Long. The String feature supports legacy code and
* should be redesigned and removed. - JH 10/12/98
*/
public long getLong(String columnID) {
    Object value = super.get(columnID);
    if (value instanceof String) {
        return Long.parseLong((String)value);
    } else {
        return ((Long)value).longValue();
    }   
}     
//---------- Private Methods -----------------------------
private static void print(String text) {
    System.out.println("Row" + text);
}

} // End class
