package org.jcon.df.request;

import org.jcon.df.work.Row;
import org.jcon.util.GenLib;
import java.sql.Statement;
import java.util.Enumeration;

/**
 * A request to update a row in an entity.
 *
 * @author Jack Harich
 */
public class UpdateRow extends DataRequest {

//---------- Private Fields ------------------------------
private Row    row;
private String entityName;
private String sql;

//---------- Abstract Implementation ---------------------
public boolean isMutator() {
    return true;
}
public Object perform() {
    // Translate sql
    try {
        // Set TStamp - DUPECODE with AddOneRow
        if (! requestServices.getTranslator()
            .getTranslatorType().isAutomaticTimeStamp()) {
            // Update TStamp manually
            String timeStamp = GenLib.createCurrentDateTime();
            this.setUpdateColumn(entityName + ".TStamp", timeStamp);
            //print(".perform() - timeStamp = " + timeStamp);
        } else {
            // Remove TStamp, since will be in update row
            this.removeUpdateColumn(entityName + ".TStamp");
        }
        sql = requestServices.getTranslator()
            .buildUpdateStm(this);

    } catch(Exception ex) {
        ex.printStackTrace();
        this.setFailureText("Cannot translate to SQL.");
        this.setFailureException(ex);
        throw new IllegalStateException("Cannot translate to SQL.");
    }
    try {

        // Translate sql
        prepare(); // To include timeStamp
        // Execute
        requestServices.getStatement().executeUpdate(sql);

    } catch(Exception ex) {
        GenLib.exception("UpdateRow.perform()",
            "Cannot update row using SQL: " + sql, ex);
        this.setFailureText("Cannot update row.");
        this.setFailureSQL(sql);
        this.setFailureException(ex);
    }
    //print(".perform() leaving: " + getState());
    return this;
}
//---------- Superclass Overrides ------------------------
public void prepare() {
    // Preconditions
    if (row == null) {
        throw new IllegalStateException("Failure to" +
        " set row.\nCannot update row without a row.");
    }
    if (entityName == null) {
        throw new IllegalStateException("Failure to" +
        " set entityName.\nCannot update row without an entityName.");
    }
}
//---------- Properties ----------------------------------
/**
 * Sets the row to be updated. This should already be
 * validated by the row's entity.
 */
public void setRow(Row newRow) {
    row = newRow;
    // Use row to add update columns
    Enumeration enum = row.getColumnIDs();
    while (enum.hasMoreElements()) {
        String columnID = (String)enum.nextElement();
        String value = (String)row.getValue(columnID);
        //print(".setRow() - " + columnID + ", '" + value + "'");
        //if (value == null) print(" - is null");
        addUpdateColumn(columnID, value);
    }
}
/**
 * Specifies the entity the row will be updated in.
 */
public void setEntityName(String entityName) {
    if (this.entityName == null) {
        addEntity(entityName);
        this.entityName = entityName;
    } else {
        GenLib.error("UpdateRow.setEntityName()",
            "EntityName already set to '" + this.entityName + "'.");
    }
}
public String getEntityName() {
    return entityName;
}
//---------- Private Methods -----------------------------
//--- Std
private static void print(String text) {
    System.out.println("UpdateRow" + text);
}

} // End class
