package org.jcon.df.request;

import java.sql.Statement;

/**
 * Represemts a request to the database for something.
 * The request contains (hopefully) no SQL and is fully
 * object oriented. Thus a request's intentions are fully
 * decoupled from the persitence mechanism.
 *
 * Request implementations are a distinct request type, such as
 * an add, remove, read, calculate, etc. The request contains
 * all the data needed to describe what's wanted.
 *
 * @author Jack Harich
 */
public interface Request extends java.io.Serializable  {


//---------- Public Methods ------------------------------
/**
 * The implementor should prepare the request using
 * available properties. Currently this is using the
 * Translator to translate before hitting the server.
 * Use of prepare is optional, since it can also be done
 * at the beginning of perform().
 * <p>
 * Throw an IllegalStateException if failure.
 */
public void prepare();

/**
 * The implementor should use the properties to perform the
 * request, placing results in itself for use by the client
 * and returning itself. Do NOT close the statement.
 * <p>
 * Implementations should return the Request, which the
 * client will cast to the Request type. We return by value,
 * since this is not a remote object.
 * <p>
 * The class implementing this interface can assume that
 * FailureText and FailureException are set to null just
 * before this method is called.
 */
public Object perform();

//---------- Properties ----------------------------------
/**
 * Returns true if the request will potentially modify data,
 * such as an add, remove or update. Otherwise returns false.
 */
public boolean isMutator();

/**
 * Returns true if request was successful, false if not.
 */
public boolean isSuccessful();

/**
 * Clears all failure data, which is FailureText,
 * FailureSQL and FailureException, by setting them to null.
 */
public void clearFailureData();

/**
 * Sets the FailureText property. This should be done to
 * indicate the request failed. Setting this property to
 * a non-null value will cause isSuccessful to be false.
 */
public void setFailureText(String failureText);

/**
 * Returns null if request was successful, otherwise returns
 * text describing the reason for failure.
 */
public String getFailureText();

/**
 * Sets the FailureSQL property. This should be done if
 * the failure had an SQL statement.
 */
public void setFailureSQL(String sql);

/**
 * Returns the FailureSQL property or null if none.
 */
public String getFailureSQL();

/**
 * Sets the FailureException property. This should be done
 * if the request failed due to an exception.
 */
public void setFailureException(Exception failureException);

/**
 * Returns the FailureException or null if none.
 */
public Exception getFailureException();

/**
 * Sets the RequestServices property. The client should
 * store this in a transient field. See RequestServices
 * for further documentation.
 */
public void setRequestServices(RequestServices services);


} // End interface
