package org.jcon.df.edit.task;

import org.jcon.df.column.*;
import org.jcon.df.Schema;
import org.jcon.param.Param;
import org.jcon.util.GenLib;
import java.util.Vector;

/**
 * Produces a Param for label/field layout.
 *
 * @author Jack Harich
 */
public class FieldLayout {

//---------- Private Fields ------------------------------
private Schema          schema;
private Vector          columnIDs;
private String          dataSourceID;

//---------- Properties ----------------------------------
public void setSchema(Schema schema) {
    this.schema = schema;
}
public void setColumnIDs(Vector columnIDs) {
    this.columnIDs = columnIDs;
}
public void setDataSourceID(String dataSourceID) {
    this.dataSourceID = dataSourceID;
}
//---------- Public Methods ------------------------------
// A default param, more properties later
// For a RowFieldPanel in first version
// Use columnIDs to create rows of Label and TextField
public Param buildParam() {
    Param param = new Param();

    // Use schema to add one row per columnID
    boolean isFirst = true;
    for (int i = 0; i < columnIDs.size(); i++) {
        String columnID = (String)columnIDs.elementAt(i);

        // Skip columns never shown to user
        if (columnID.endsWith(".MID")) continue;
        if (columnID.endsWith(".TStamp")) continue;
        if (columnID.endsWith(".LastUserSysMID")) continue;

        //print(" - Creating row for columnID " + columnID);

        ColumnDef def = schema.getColumnDef(columnID);
        addRow(param, columnID, def, isFirst);
        isFirst = false;

        // Debug - rows in wrong order - Fixed
        //print(".buildParam() - DataKeys: "
        //    + param.getDelimitedDataKeys());
    }
    return param;
}
private void addRow(Param param, String columnID,
        ColumnDef def, boolean isFirst) {

    //print(".addRow() - Adding " + columnID);
    String name = def.getName();
    // Label
    String displayName = def.getDisplayName();
    addLabelField(param, name, displayName);

    // Field *** Add more types as needed
    String type = def.getType();

    if (type == "MID") {
        addLinkField(param, columnID, def, isFirst);

    } else if (type == "Boolean") {
        addBooleanField(param, columnID, def, isFirst);

    } else if (type == "Memo") {
        addMemoField(param, columnID, def, isFirst);

    } else {
        addTextField(param, columnID, def, isFirst);
    }
}
private void addLabelField(Param param, String prefix,
        String displayName) {

    String key = prefix + "Label.";
    param.put(key + "Type", "Label");
    param.put(key + "Style", "FieldRight");
    param.put(key + "Text", displayName);
}
/** Data format - For UserAdmin.SysUserMID link column
 * DataSourceID is: Source1...
 * EntityName is: SysUser
 * ColumnNames is: LastName, FirstName
 */
private void addLinkField(Param param, String columnID,
        ColumnDef def, boolean isFirst) {

    String key = def.getName() + "Field.";
    param.put(key + "Type",    "Link");
    param.put(key + "Style",   "FieldLeft");
    param.put(key + "AddType", "EndRow");

    param.put(key + "DataSourceID",  dataSourceID);

    MIDColumn midColumn = (MIDColumn)def;

    //String entityName = prop.getPrimeEntityName();
    String entityName = midColumn.getLinkEntityName();
    param.put(key + "EntityName", entityName);

    String desColNames = schema.getEntity(entityName).getDescriptionColumnNames();
    if (desColNames == null) {
        GenLib.error("FieldLayout.addLinkFielf()",
            "DescriptionColumnNames is missing for entity " + entityName);
    }
    param.put(key + "ColumnNames", desColNames);

    param.put(key + "ColumnID", columnID);
    if (isFirst) param.put(key + "FirstComponent", "true");
    //print(".addLinkField() - param \n" + param);
}
private void addBooleanField(Param param, String columnID,
        ColumnDef def, boolean isFirst) {

    String key = def.getName() + "Field.";
    param.put(key + "Type",    "Boolean");
    param.put(key + "Style",   "FieldLeft");
    param.put(key + "AddType", "EndRow");

    param.put(key + "ColumnID", columnID);
    if (isFirst) param.put(key + "FirstComponent", "true");
}
private void addMemoField(Param param, String columnID,
        ColumnDef def, boolean isFirst) {

    MemoColumn memoDef = (MemoColumn)def;
    int rows    = memoDef.getPreferredRows();
    int columns = memoDef.getPreferredColumns();

    String key = def.getName() + "Field.";
    param.put(key + "Type",    "Memo");
    param.put(key + "Style",   "FieldLeft");
    param.put(key + "AddType", "EndRow");

    param.put(key + "Rows",     rows + "");
    param.put(key + "Columns",  columns + "");
    param.put(key + "ColumnID", columnID);
    if (isFirst) param.put(key + "FirstComponent", "true");
}
private void addTextField(Param param, String columnID,
        ColumnDef def, boolean isFirst) {
    // Calc columns
    int length = def.getPreferredLength();
    if (length < 2) length = 2; // One is hard on user
    String columns = String.valueOf(length);

    String key = def.getName() + "Field.";
    param.put(key + "Type",    "TextField");
    param.put(key + "Style",   "FieldLeft");
    param.put(key + "AddType", "EndRow");

    param.put(key + "Columns",  columns);
    param.put(key + "ColumnID", columnID);
    if (isFirst) param.put(key + "FirstComponent", "true");
}
//--- Std
private static void print(String text) {
    System.out.println("FieldLayout" + text);
}

} // End class
