package org.jcon.df.edit.control.tree;

import org.jcon.df.edit.control.element.TreeElement;
import org.jcon.ui.ImageLib;
import org.jcon.ui.VisualLib;
import org.jcon.util.GenLib;
import org.jcon.util.TNode;
import java.awt.*;

import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JTree;
import javax.swing.tree.TreeCellRenderer;
import javax.swing.tree.DefaultMutableTreeNode;

/**
* Renders tree nodes using the node userObject, which is
* an Element. The methods Element.getIcon() and Element.
* getText() are relied upon. 
* 
* @author Jack Harich
*/
public class TreeControlCellRenderer
    extends JLabel implements TreeCellRenderer {

//---------- Protected Fields ----------------------------
// These are the component's state, used for paint()
protected boolean     selected;
protected TreeElement element;
protected Font        font; // null unless custom

//---------- TreeCellRenderer Implementation -------------
/**
 * This is messaged from JTree whenever it needs to get the
 * size of the component or it wants to draw it.
 */
public Component getTreeCellRendererComponent(JTree tree,
            Object value, boolean selected,
            boolean expanded, boolean leaf, int row,
            boolean hasFocus) {

    // Set the text. Used by tree to determine row length.
    element = (TreeElement) ((TNode)value).getUserObject();
    super.setText(element.getText());

    // Set image for node icon
    if(expanded) {
        super.setIcon(element.getExpandedIcon());
    } else {
        super.setIcon(element.getCollapsedIcon());
    }
    // Update the selected flag for the next paint
    this.selected = selected;
    
    // Prepare font. We could optimize more here.
    if (element.isCustomFont()) {
        font = tree.getFont();
        
        if (element.isBoldFont()) {
            font = VisualLib.makeFontBold(font);
            super.setFont(font);
        }
        if (element.getExtraFontSize() != 0) {
            font = VisualLib.createLargerFont(font, element.getExtraFontSize());
            super.setFont(font);    
        }
    } else {
        font = null;
    }
    // Done
    return this;
}
//---------- Superclass Overrides ------------------------
/**
 * paint() is subclassed to draw the background correctly.
 * JLabel currently does not allow backgrounds other than
 * white, and it will also fill behind the icon,
 * something that isn't desirable.
 */
public void paint(Graphics g) {
    // Set background color to selected or normal
    if (selected) {
        g.setColor(Color.cyan);  // Could be a constant
    } else {
        g.setColor(Color.white); // Could be a constant
    }
    // Prep
    // Gap over 3 pixels causes right text clipping
    // To get larger gap, put extra left transparency on icon
    // ( should use property iconTextGap ?)
    // *** int         GAP = 3; // WE PAINT  Icon, gap, text
    
    // Calculate iconWidth including gap
    int   iconWidth = 0; // Icon, gap
    Icon  currentIcon = super.getIcon();
    if(currentIcon != null) {
        iconWidth = currentIcon.getIconWidth() + super.getIconTextGap();
    }
    //----- Draw icon
    if (currentIcon != null) {
        currentIcon.paintIcon(this, g, 0, 2); // x, y
    }
    //----- Calculations for bkg and text
    if (font != null) g.setFont(font);
    String      text = super.getText();
    FontMetrics metrics = g.getFontMetrics();
    int         lineHeight = getHeight() - 1;
   
    int yAscent = metrics.getAscent() + 1;
    int textHeight = metrics.getHeight();
    int y = ((lineHeight - textHeight) / 2) + yAscent;
        
    //----- Draw cell background color rectangle
    // This is bkg or selected color
    int textWidth = metrics.stringWidth(text);
    // was g.fillRect(iconWidth, 1, textWidth, lineHeight);
    g.fillRect(iconWidth, y - yAscent + 1, textWidth, textHeight);

    //----- Draw text, centered vertically
    g.setColor(Color.black); // Foreground
    g.drawString(text, iconWidth, y);
}
// From swing BasicTreeCellRenderer. Optimized.
public Dimension getPreferredSize() {
    Dimension dim = super.getPreferredSize();

    if(dim != null) {
        // Extra width is Swing bug fix from BasicTreeCellRenderer, 3 extra pixels
        if (super.getIcon() != null) {
            // Without extra height, icon bottom is clipped
            // This is an apparent Swing bug fix
            return new Dimension(dim.width + 3, dim.height + 2);
        } else {
            return new Dimension(dim.width + 3, dim.height);
        }
    }
    return dim;
}
//---------- Private Methods -----------------------------
//--- Std
private static void print(String text) {
    System.out.println("TreeControlCellRenderer" + text);
}


} // End class
