package org.jcon.df.edit.control;

import org.jcon.util.GenLib;
import org.jcon.df.work.Row;
import java.awt.Component;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

/**
 * Manages a group of controls. Its most important
 * responsibility is to be able to set and get a Row,
 * using its bound controls.
 * <p>
 * Within each group the control ColumnIDs must be
 * unique. This allows rows to be easily handled.
 *
 * @author Jack Harich
 */
public class ControlGroup {

//---------- Private Fields ------------------------------
private Vector    controls = new Vector();
private Component firstComponent; // null if none

// Key = ColumnID, Object = Control
private Hashtable boundControls = new Hashtable();

// *** Later namedControls with name as Key

//---------- Initialization ------------------------------

//---------- Properties ----------------------------------
public void setControls(Vector controls) {
    this.controls = controls;
    // Refresh boundControls
    firstComponent = null;
    boundControls.clear();
    for (int i = 0; i < controls.size(); i++) {
        Control control = (Control)controls.elementAt(i);
        String ColumnID = control.getColumnID();
        //print(".setControls() - " + control.getName() + ", ColumnID = " + ColumnID);
        if (ColumnID != null) {
            boundControls.put(ColumnID, control);
        }
        if (control.isFirstComponent()) {
            firstComponent = control.getComponent();
        }
    }
}
public Vector getControls() {
    return controls;
}
//---------- Public Methods ------------------------------
/**
 * For each row element, uses the ColumnID to set the
 * bound control's value. Ignores ColumnIDs not in controls.
 * If row is null then all bound controls are set to null.
 * Returns null for success.
 */
public String setRow(Row row) {
    if (row == null) {
        emptyRow();
        return null; // Okay
    }
    Enumeration enum = row.getColumnIDs();
    while (enum.hasMoreElements()) {
        String columnID = (String)enum.nextElement();
        //print(" checking " + columnID);
        Control control = (Control)boundControls.get(columnID);
        if (control == null) continue;
        try {
            //print(".setRow() - Setting " + columnID);
            control.setValue(row.getValue(columnID));
        } catch(Exception ex) {
            String problem = "Can't set columnID " + columnID;
            GenLib.exception("ControlGroup.setRow()",
                problem, ex);
            return problem;
        }
    }
    return null; // Okay
}
/**
 * Sets all bound controls to null.
 */
public void emptyRow() {
    Enumeration enum = boundControls.elements();
    while (enum.hasMoreElements()) {
        Control control = (Control)enum.nextElement();
        control.setValue(null);
        //print(".emptyRow() - setting to null " + control.getName());
    }
}
/**
 * Returns a new row containing all bound controls.
 */
public Row getRow() {
    Row row = new Row();
    Enumeration enum = boundControls.elements();
    while (enum.hasMoreElements()) {
        Control control = (Control)enum.nextElement();
        //print(".getRow() - columnID = " + control.getColumnID());
        row.setValue(control.getColumnID(), control.getValue());
    }
    return row;
}
/**
 * Sets the Editable property of all bound controls. Useful
 * for making data uneditable, such as if only viewing it.
 */
public void setEditable(boolean editable) {
    Enumeration enum = boundControls.elements();
    while (enum.hasMoreElements()) {
        Control control = (Control)enum.nextElement();
        if (control.isEditAllowed() && editable) {
            control.setEditable(true);
        } else {
            control.setEditable(false);
        }
    }
}
/**
 * Puts the focus on the control with the "First"
 * property = true. If none have this property = true
 * then there is no effect;
 */
public void requestFocus() {
    if (firstComponent != null) firstComponent.requestFocus();
}
/**
 * Puts the focus on the bound control with the columnID.
 * If not found there is no effect.
 */
public void requestFocus(String columnID) {
    Enumeration enum = boundControls.elements();
    while (enum.hasMoreElements()) {
        Control control = (Control)enum.nextElement();
        if (control.getColumnID().equals(columnID)) {
            control.getComponent().requestFocus();
            return;
        }
    }
}
//---------- Private Methods -----------------------------
//--- Std
private static void print(String text) {
    System.out.println("ControlGroup" + text);
}

} // End class
