package org.jcon.df.edit.control;

import org.jcon.df.edit.EditListener;
import org.jcon.df.edit.widget.Widget;
import org.jcon.df.column.ColumnDef;
import org.jcon.param.Param;
import java.awt.Component;

/**
 * The superclass for named components, each of which is a
 * visual "control". Some are merely for general
 * layout, such as fancy borders. Some present fixed visuals
 * such as labels. Some are "bound", having viewable and
 * often editable data. Bound controls have a DataSourceID
 * and ColumnID.
 * <p>
 * The subclasses are wrappers for their Component. Bound
 * subclasses should override setValue() and getValue().
 * Most subclasses should override init().
 * <p>
 * The following properties are important: <pre>
 *
 * Name - Descriptive, unique per ControlGroup
 *   Can be used to lookup controls
 * Param - The Param the control used to config itself
 * ColumnID - Not null if bound
 * ColumnDef - null if none  </pre>
 *
 * @author Jack Harich
 */
public abstract class Control implements Widget {

//---------- Private Fields ------------------------------
protected String    name;
protected String    columnID;
protected String    helpID = null;
protected boolean   isFirstComponent;
protected boolean   editAllowed = true; // Default

protected Param        myParam;
protected ColumnDef    columnDef;
protected EditListener editListener;

//---------- Widget Implementatation ---------------------
//----- name
public void setName(String name) {
    this.name = name;
}
public String getName() {
    return name;
}
//----- Other
public void setParam(Param param) {
    myParam = param;
    isFirstComponent = param.isTrueDefaultFalse("FirstComponent");
    columnID = param.getString("ColumnID");
}
public void setEditListener(EditListener editListener) {
    this.editListener = editListener;
}
public void requestFocus() {
    // Do nothing. This was added very late, 2/7/99 JH
    // The first use was on HyperlinkMenuControl to
    // allow showing InfoText when control is shown
}
//----- abstract
public abstract Component getComponent();

//---------- Properties ----------------------------------
//----- param
public Param getParam() {
    return myParam;
}
//----- isFirstComponent
public void setFirstComponent(boolean isFirst) {
    isFirstComponent = isFirst;
}
public boolean isFirstComponent() {
    return isFirstComponent;
}
//----- editAllowed
/**
 * Only controlIDs in the prime entity may be editable,
 * and even some of these may not be editable in some tasks.
 * ControlIDs not in the prime entity should never be
 * editable. Please note the default is true for prime
 * entity columns.
 */
public void setEditAllowed(boolean editAllowed) {
    this.editAllowed = editAllowed;
}
public boolean isEditAllowed() {
    return editAllowed;
}
//----- columnID
public void setColumnID(String columnID) {
    this.columnID = columnID;
}
public String getColumnID() { // null if not bound
    return columnID;
}
//----- helpID
public void setHelpId(String helpID) {
    this.helpID = helpID;
}
public String getHelpId() { // null if not bound
    return helpID;
}
//----- Other - Setters
public void setColumnDef(ColumnDef columnDef) {
    this.columnDef = columnDef;
}
public void setEditable(boolean editable) {
    // Do nothing
}
//---------- Public Methods ------------------------------
// Override these methods to implement
public void init() { // Called after init setters
    // Do nothing
}
public void setValue(Object value) {
    // Do nothing,
}
public Object getValue() {
    return null; // Do nothing
}
//---------- Private Methods -----------------------------
//--- Std
private static void print(String text) {
    System.out.println("Control" + text);
}

} // End interface
