package org.jcon.df.column;

import org.jcon.util.GenLib;
import org.jcon.param.Param;

/**
 * A Java String of minimum and maximum length. There are
 * no content restrictions. The defaults for minLength and
 * maxLength are 0 and 50.
 *
 * This class represents an empty value as "".
 *
 * @author Jack Harich
 */

public class StringColumn extends ColumnDef {

//---------- Private Fields ------------------------------
private int minLength = 0;
private int maxLength = 50;
private boolean autotrim = true;

//---------- Initialization ------------------------------
/**
 * No-arg to allow serialization.
 */
public StringColumn() {
    super();
    setDefaultValue("");
    type = "String";
}
/**
 * A convience constructor. Also sets default to "".
 */
public StringColumn(String name, int maxLength) {
    this();
    setName(name);
    setMaxLength(maxLength);
}

//---------- Abstract Implementation ---------------------
public void initFromParam(Param param) {
    checkStandardProperties(param); // superclass
    if (param.hasProperty("MaxLength")) {
        setMaxLength(param.getInt("MaxLength"));
    }
}
/**
 * Checks for required, must be a String, minLength and
 * maxLength.
 */
public String validateString(String text) {

    if (text.length() < minLength) {
        return displayName + " must be at least " + minLength +
            " characters long.";

    } else if (text.length() > maxLength) {
        return displayName + " cannot be over " + maxLength +
            " characters long.";

    } else {
        return null;
    }
}
public String getMyPropertySummary() {
    String summary = "";
    if (minLength != 0) summary += "Min=" + minLength + ",";
    summary += "Max=" + maxLength + ",";
    return summary;
}
//---------- Superclass Overrides ------------------------
public int getPreferredLength() {
    // *** May change
    if (maxLength > 50) {
        return 50;
    } else {
        return maxLength;
    }
}
//---------- Public Methods ------------------------------
/**
 * The default for minimum length is zero. There is no
 * maximum.
 */
public void setMinLength(int min) {
    if (min < 0 ) {
        GenLib.error("StringColumn.setMinLength()",
            "Minimum length cannot be less than zero.");
    } else {
        this.minLength = min;
    }
}
public int getMinLength() {
    return minLength;
}
/**
 * The default for maximum length is 50. There is no
 * maximum. The minimum is zero. Though zero makes little
 * sense, it allows a field to be designed and implemented
 * but not used.
 */
 public void setMaxLength(int max) {
    if (max < 0 ) {
        GenLib.error("StringField.setMaxLength()",
            "Maximum length cannot be less than zero.");
    } else {
        this.maxLength = max;
    }
}
public int getMaxLength() {
    return maxLength;
}
/**
 * If autotrim is true then whenever the value is set it
 * will be trimmed of leading and trailing spaces. The
 * default is true, which is highly recommended.
 */
public void setAutotrim(boolean autotrim) {
    this.autotrim = autotrim;
}
public boolean isAutotrim() {
    return autotrim;
}
//---------- Private Methods -----------------------------
private void print(String text) {
    System.out.println(text);
}

} // End class
