package org.jcon.df.column;

import org.jcon.util.GenLib;
import org.jcon.param.Param;

/**
 * A MaskColumn value must satisfy a mask. For example
 * an SSN would have a mask of "999-99-9999".
 * Currently only "9" and "-" mask characters are supported.
 *
 * This class represents an empty value as "".
 *
 * @author Jack Harich
 */

public class MaskColumn extends ColumnDef {

//---------- Private Fields ------------------------------
private String validMask;

//---------- Initialization ------------------------------
/**
 * No-arg to allow serialization.
 */
public MaskColumn() {
    super();
    setDefaultValue("");
    type = "Mask";
}
/**
 * A convienence constructor.
 */
public MaskColumn(String name, String validMask) {
    this();
    setName(name);
    this.validMask = validMask;
}
public static void main(String args[]) { // Unit test
    MaskColumn test = new MaskColumn();
    test.runUnitTest();
}
//---------- Abstract Implementation ---------------------
public void initFromParam(Param param) {
    checkStandardProperties(param); // superclass
    validMask = param.getString("Mask"); // Required
}
/**
 * Checks that the value entered satisfies the mask. Returns
 * null if okay or a message explaining the problem.
 */
public String validateString(String text) {
    if (validateMask(validMask, text) == true) {
        return null;
    } else {
    return displayName + " must be of the format '" +
        validMask + "'.";
    }
}
public String getMyPropertySummary() {
    return "Mask=" + validMask + ",";
}
//---------- Superclass Overrides ------------------------
public int getPreferredLength() {
    return validMask.length() + 1;
}
protected String getAutomaticDescription() {
    return "This must be of the format as " + validMask + ".";
}
//---------- Private Methods -----------------------------
public boolean validateMask(String mask, String text) {
    // Check length
    if (mask.length() != text.length() ) return false;

    char DIGIT = '9';
    char DASH  = '-';
    for (int i = 0; i < mask.length(); i++) {

        char textChar = text.charAt(i);
        char maskChar = mask.charAt(i);

        if (maskChar == DIGIT) {
            if (! Character.isDigit(textChar)) return false;

        } else if (maskChar == DASH) {
            if (textChar != DASH) return false;

        } else {
            GenLib.error("MaskColumn.validateMask()",
                "Unknown mask character in " + mask);
            return false;
        }
    }
    return true;
}
private void testMask(String mask, String text) {
    if (validateMask(mask, text) == true) {
        print("'" + text + "' okay");
    } else {
        print("'" + text + "' invalid");
    }
}
private void runUnitTest() {
    String mask =  "999-99-9999";

    testMask(mask, "123-45-6789");
    testMask(mask, "123-XX-4567");
    testMask(mask, "-123-99-4567");
    testMask(mask, "123456789");
    testMask(mask, "345-9");
    testMask(mask, "");
    testMask(mask, "0");
    testMask("999-X9-9999", "999-X9-9999");

    print("Test complete");
}
//--- Std
private void print(String text) {
    System.out.println(text);
}

} // End class
