package org.jcon.df.column;

import org.jcon.param.Param;

/**
 * A Java Integer of minimum and maximum value. The
 * defaults for minValue and maxValue are 0 and 1,000,000.
 * The defaultValue is zero, just as in Java.
 *
 * Integers range from -2,147,483,648 to 2,147,483,647.
 * An Integer cannot include a decimal point or a "+".
 *
 * @author Jack Harich
 */

public class IntegerColumn extends ColumnDef {

//---------- Protected Fields ----------------------------
protected int minValue = 0;
protected int maxValue = 1000000; // 1,000,000

//---------- Initialization ------------------------------
/**
 * No-arg to allow serialization.
 */
public IntegerColumn() {
    super();
    setDefaultValue("0");
    type = "Integer";
}
/**
 * A convience constructor.
 */
public IntegerColumn(String name) {
    this();
    setName(name);
}
/**
 * Another convience constructor.
 */
public IntegerColumn(String name, int minValue, int maxValue) {
    this(name);
    this.minValue = minValue;
    this.maxValue = maxValue;
}
public static void main(String args[]) {
    IntegerColumn test = new IntegerColumn();
    test.runUnitTest(args);
}
//---------- Abstract Implementation ---------------------
public void initFromParam(Param param) {
    checkStandardProperties(param); // superclass
    if (param.hasProperty("MinValue")) {
        setMinValue(param.getInt("MinValue"));
    }
    if (param.hasProperty("MaxValue")) {
        setMaxValue(param.getInt("MaxValue"));
    }
}
/**
 * Checks for required, must be an Integer, minValue and
 * maxValue.
 */
public String validateString(String value) {
    int number = 0;
    // Check to see if int
    try {
        number = Integer.parseInt(value);
    } catch(Exception ex) {
        return displayName + " is not an Integer.";
    }
    // Check min and max
    if (number < minValue || number > maxValue) {
        return displayName + " must be between " +
            minValue + " and " + maxValue;
    } else {
        return null; // Okay
    }
}
public String getMyPropertySummary() {
    String summary = "";
    summary += "Min=" + minValue + ",";
    summary += "Max=" + maxValue + ",";
    return summary;
}
//---------- Superclass Overrides ------------------------
public int getPreferredLength() {
    int length = String.valueOf(maxValue).length() + 1;
    if (length < 3) length = 3;
    return length;
}
protected String getAutomaticDescription() {
    return "Must be an integer between " + minValue
        + " and " + maxValue + ".";
}
//---------- Public Methods ------------------------------
//--- Pair
public void setMinValue(int minValue) {
    this.minValue = minValue;
}
public int getMinValue() {
    return minValue;
}
//--- Pair
public void setMaxValue(int maxValue) {
    this.maxValue = maxValue;
}
public int getMaxValue() {
    return maxValue;
}
//---------- Private Methods -----------------------------
private void runUnitTest(String args[]) {
    if (args.length == 0) {
        print("Enter test values after className.");
        return;
    }
    int min = -10;
    int max = 1000;
    String value, message;
    IntegerColumn number = new IntegerColumn("TestInteger");
    number.setMinValue(min);
    number.setMaxValue(max);
    for (int i = 0; i < args.length; i++) {
        value = args[i];
        message = number.validate(value);
        if (message == null) {
            print(value + " is okay.");
        } else {
            print(value + "  " + message);
        }
    }
    print("Test complete with min/max of " + min + "/" + max );
}
//--- Std
private void print(String text) {
    System.out.println(text);
}

} // End class
