package org.jcon.core;

import java.util.Enumeration;

/**
* This interface represents a cell (aka container) in a 
* System Tree. A cell holds zero or more other parts, each
* of which may be a cell or leaf. Since Cell extends Part,
* we are using the Composite Pattern. 
* <p>
* Responsiblities are to manage the collection of parts in
* the cell and their relationships. 
* <p>
* The design philosophy is to delegate as much work as
* practical to other core parts, which are in the cell
* or are a core service. This eliminates a separate 
* "kernel", allows easy core customization, and helps
* achieve the goal of Infinite Extensibility. 
* <p>
* We have named this "Cell" rather than "Container" to 
* better articulate that we are basing our behavior model 
* on biology. In a System Tree one Cell can hold another
* Cell, so the analogy is not exact. Cells are the basic
* building blocks of life, have internal  parts, use DK 
* (DNA and RNA), use Messages, and use pumps for object 
* flow. Software has much to learn from mother nature....
* <p>
* Unfortunately this interface may become large and complex.
* This is to be expected due to its nature, but can be
* minimized through good design and ample delegation.
* <p>
* Note that the various part getters include hidden parts.
* These should not be exposed to domain parts, so we will
* provide something like a CellServices fascade. TODO
* <p>
* TODO - Many features are not yet supported, such as 
* companion part, Messages, DK, worker special behavior,
* full cell lifecycle, 
*
* @author Jack Harich
*/
public interface Cell extends Part, StartablePart,
    Replicable {

/**
* Adds the part to the cell. The part's name must
* be unique to the cell and not already in it.
* Throws an IllegalStateException if the part's name is
* already in the cell. The part's parent is set to
* this cell.
*/
public void addPart(Part part);

/**
* Removes the part from the cell. Returns the part
* if removed or null if not found. If found, the part's
* parent is set to null.
*/
public Part removePart(String partName);

/**
* Returns the named part in this cell or null if not
* found. Plese note this searches only this cell and
* may return null. Ability to get lower part not supported.
*/
public Part getPart(String partName);

/**
* Returns an Enumeration of the parts in this cell, in the
* order in which they were added. This better preserves the
* designer's organization, such as what appears in a
* visual System Tree.
*/
public Enumeration getParts();

/**
* Returns the worker instance for the partName, finding it
* in a manner identical to CellServices.findPart(partName).
* The partName must be a worker, not a cell. This method
* is so useful it's here as well as in CellServices.
*/
public Object findWorker(String partName);

/**
* The cell should initialize itself. It can assume it has
* all necesary core parts and that parent cells have
* been inititalized. 
*/
public void initCell();


} // End interface