package org.jcon.ba.system.router;

import java.util.EventObject;

/**
 * The superclass for EventRouter listener proxies, aka
 * proxy. The proxy allows the EventRouter to dispatch
 * an event to those who have subscribed to the router for
 * that event.
 *
 * This class has two main responsibilities: serving as a
 * proxy and dispatching an event to a listener. We could
 * have had separate proxy and dispatcher classes, but this
 * approach has higher cohesion.
 *
 * The subclass must implement an appropriate event
 * interface, such as ActionListener. In the interface
 * implementation the subclass should set its event field
 * BEFORE calling the eventRouter.
 *
 * @author Jack Harich
 */
public abstract class EventListenerProxy {

//---------- Protected Fields ----------------------------
protected EventRouter eventRouter;
protected Object event;

//---------- Properties ----------------------------------
public void setEventRouter(EventRouter eventRouter) {
    this.eventRouter = eventRouter;
}
/**
 * Not used by the event system, but provided for
 * completeness and symmetry.
 */
public EventRouter getEventRouter() {
    return eventRouter;
}
/**
 * In the event interface implementation the subclass
 * must set the event field BEFORE calling eventRouter.
 */
public Object getEvent() {
    return event;
}
//---------- Abstract Methods ----------------------------
/**
 * The proxy should add itself to the event source.
 */
abstract public void addToSource(Object source);

/**
 * The proxy should remove itself from the event source.
 */
abstract public void removeFromSource(Object source);

/**
 * The subclass should dispatch the event to
 * the @param listener. This will involve casting @param
 * listener, and calling the appropriate listener method.
 *
 * In cases where an event could be sent to more than one
 * listener method, such as a WindowEvent, this method
 * should make that distinction using the event.
 */
abstract public void notifyListener(Object listener);

} // End class
