package org.jcon.ba.system;

import org.jcon.param.ParamStore;

/**
 * An optional interface for ItemSystem class markers to
 * implement to customize the ItemSystem. If this interface
 * is not used then the ItemSystem uses various defaults.
 *
 * If the class is not the root the interface is ignored.
 * Thus to reuse the system represented by this class in a
 * larger system the ItemDef must be spedified properly in
 * the _Container.parex, and the class should implement
 * ParamDriven if it has parameters. This is conceptually
 * awkward, but does let the class be both a root and a
 * reused bean. A system reuse example is:
 *
 * textEditor_Item has:
 *     IsContainer is: true
 *     BeanClassName is: org.jcon.util.TextEditor
 *     ParamMarkerItemName is: this
 *     ParamFileName is: TextEditor.parex
 *     End: textEditor_Item
 *
 * It's possible that the root bean will optionally provide
 * properties for BA behavior in the future, in addition to
 * application specific system behavior. ***
 * For example it might provide:
 * - SystemDisplayName
 * - Initial tree position or opened state
 * - Plugins such as html editor, configuration mgt,
 *      customer customizer. custom edit param, etc
 *
 * @author Jack Harich
 */
public interface ItemSystemMarker {

/**
 * Returns the ParamStore to be used in this ItemSystem or
 * null to used the default. This allows a custom Param
 * storage mechanism to be used per system.
 */
public ParamStore getParamStore();

/**
 * Returns the fully qualified class name of the root bean,
 * or null if none desired.
 * This is the only way to provide a root bean, since
 * it cannot be specified in a _Container.parex.
 *
 * A huge advantage of root beans is they can singlehandly
 * provide the parameters to customize an entire system,
 * without having to change anything in the system's nodes.
 * One example of this is setting the root container's
 * PropMap via BasketServices, using their Param.
 * They can also provide custom behavior independently of
 * their parameters.
 *
 * This is often the same as the marker class, for high
 * cohesion. In this case the marker and bean are the same.
 */
public String getBeanClassName();

/**
 * Returns the fully qualified class name used to locate
 * the Param or returns null if no Param is used or the
 * BeanClassName should be used. We cannot use item name
 * here since the root bean behaves differently.
 *
 * This is often the same as the BeanClassName.
 */
public String getParamMarkerClassName();

/**
 * Returns the param file name containing the Param text or
 * null. This is typically null or something like
 * "_CustomerA.parex".
 */
public String getParamFileName();


} // End interface


/**

import org.jcon.ba.system.ItemSystemMarker;
import org.jcon.param.ParamStore;

//---------- ItemSystemMarker Implementation -------------
public ParamStore getParamStore() {
    return null; // Default
}
public String getBeanClassName() {
    return this.getClass().getName();
}
public String getParamMarkerClassName() {
    return null; // None or default
}
public String getParamFileName() {
    return null; // None or default
}

*/
