package org.jcon.ba.system;

import org.jcon.param.Param;
import org.jcon.param.ParamDriven;
import org.jcon.util.service.ContainerServices;
import org.jcon.util.service.ContainerServicesUser;

/**
 * A wrapper for a standard Java bean. This allows any
 * Java bean to be used in the BA without modification,
 * except in the rare cases of method name conflicts.
 *
 * We provide a flexible aproach to bean reuse and integration
 * with the BA. The bean instance may optionally implement: <BR>
 * - BeanAction*, for various reasons. <BR>
 * - BeanPriv, to use BasketServices. <BR>
 * - ParamDriven to have it's Param provided. <BR>
 * - MessageListener to receive events. <BR>
 * - MessageSource to fire events. (or extend MessageSourceStd) <BR>
 * - ContainerServicesUser to receive its Basket's ContainerServices
 *
 * Most beans will implement MessageListener, many will extend
 * MessageSourceStd, and some will implement the various
 * BeanAction* interfaces, especially BeanActionStart.
 *
 * NOTE - BeanAction is reserved for future use, such as
 * the interface that the BeanAction* interfaces extend.
 *
 * @author Jack Harich
 */
public class BeanWrapper implements java.io.Serializable {

//---------- Private Fields ------------------------------
private Object  instance; // The bean
private boolean instanceStarted;
private int     myState = Item.CREATED;

//---------- Properties ----------------------------------
public Object getInstance() {
    return instance;
}
public boolean isParamDriven() {
    return ((instance instanceof ParamDriven) ? true : false);
}
// The "true" state considering interfaces
public int getTrueState() {
    return myState;
}
//---------- Package Methods -----------------------------
//----- The "Big 6"
void init() {
    if (instance instanceof BeanActionInit) {
        ((BeanActionInit)instance).init();
        myState = Item.INITIALIZED;
    }
}
void start() {
    if (! instanceStarted) init();

    if (instance instanceof BeanActionStart) {
        ((BeanActionStart)instance).start();
        myState = Item.RUNNING;
    }
    instanceStarted = true;
}
void pause() {
    if (instance instanceof BeanActionPause) {
        ((BeanActionPause)instance).pause();
        myState = Item.PAUSED;
    }
}
void resume() {
    if (instance instanceof BeanActionPause) {
        ((BeanActionPause)instance).resume();
        myState = Item.RUNNING;
    }
}
String canClose() {
    if (instance instanceof BeanActionClose) {
        return ((BeanActionClose)instance).canClose();
    } else {
        return null; // This means yes I can close
    }
}
void close() {
    if (instance instanceof BeanActionClose) {
        ((BeanActionClose)instance).close();
        myState = Item.CREATED;
    }
    instanceStarted = false;
}
//----- Other
void setInstance(Object instance, BasketServices services) {
    this.instance = instance;
    instanceStarted = false;
    myState = Item.CREATED;

    // Optionally provide BasketServices to privileged beans
    if (instance instanceof BeanPriv) {
        ((BeanPriv)instance).setBasketServices(services);
    }
    // Optionally provide ContainerServices
    if (instance instanceof ContainerServicesUser) {
        ((ContainerServicesUser)instance)
            .setContainerServices(services.getContainerServices());
    }
}
//---------- Private Methods -----------------------------
//--- Std
private static void print(String text) {
    System.out.println("BeanWrapper" + text);
}

} // End class
