package org.jcon.ba.lib;

import org.jcon.ba.lib.LocaleMgr;
import org.jcon.ba.system.*;
import org.jcon.ui.MessageBox;
import org.jcon.util.GenLib;
import org.jcon.util.msg.Message;
import org.jcon.util.msg.MessageListener;

/**
 * Handles close system requests in a reusable manner.
 * Listens for the close command. When received it:
 * - Check to see if the system can close.
 * - If not, it explains why and ends
 * - If yes then the system is closed correctly
 *     and if exitAfterClose is true
 *     System.exit(0) is called.
 *
 * This is a good example of a reusable BA business object
 * with a single, well chosen responsibility.
 *
 * @author Jack Harich
 */
public class SystemCloser implements BeanPriv, MessageListener {

//---------- Private Fields ------------------------------
private BasketServices basketServices;
private String  closeCommand = "CloseSystem";
private boolean exitAfterClose = false;
private LocaleMgr localeMgr = null; // CMC - Added

//---------- MessageListener Implementation --------------
public void processMessage(Message message) {
    String name = message.getName();
    //print(".processMessage() - name = '" + name + "'");

    if (name.equals(closeCommand)) {
        exitSystemRequested();
    }
}
public String[] loadMessageInterests() {
    return new String[] {closeCommand};
}
//---------- BeanPriv Implementation ---------------------
public void setBasketServices(BasketServices services) {
    basketServices = services;
}
//---------- Properties ----------------------------------
//----- closeCommand
/**
 * Sets the close command this class listens for. The
 * default is "CloseSystem".
 */

public void setCloseCommand(String closeCommand) {
    this.closeCommand = closeCommand;
}
public String getCloseCommand() {
    return closeCommand;
}
//----- exitAfterClose
/**
 * Sets the boolean determining whether System.exit(0) will
 * be called after the system is closed. The default is false.
 */
public void setExitAfterClose(boolean exitAfterClose) {
    this.exitAfterClose = exitAfterClose;
}
public boolean isExitAfterClose() {
    return exitAfterClose;
}
/**
* This property is used to localize button labels and such
* that are not user controlled. If not supplied or null
* then the default locale is used. All other text must be
* localized by the user of this class.
*/
public void setLocaleMgr(LocaleMgr localeMgr) {
	this.localeMgr = localeMgr;
}
//---------- Private Methods -----------------------------
private void exitSystemRequested() {
    // Can the root, ie entire system, be closed?

    // CMC - this reason must be internationalized.
    String reason = basketServices.canCloseItem("this");
    if (reason != null) {
        GenLib.helpfulHint(getTextMessage("CannotCloseMsg")
            + "\n" + reason);
        return;
    }
    // Confirm
    MessageBox box = new MessageBox();
    String prefix = (exitAfterClose ? getText("Exit") : getText("Close")); // CMC - Added getText()
    // CMC - Added localization call if localeMgr is specified
    box.setTitle(prefix + " " + getTextMessage("SystemConfirmation"));
    box.addChoice(getText("Yes"), "Yes"); // CMC Added getText and command
    box.addChoice(getText("No"), "No");   // CMC Added getText and command
    box.setCloseWindowCommand("No");
    String answer = box.askBlock(getTextMessage("ProceedToExit"));

    if (answer == "Yes") {
        basketServices.closeItem("this");
        System.exit(0);
    }
}
private String getText(String textName) {
    // CMC - Added
    if (localeMgr == null) {
        return textName;
    } else {
    return localeMgr.getString("UtilitiesJCON",
            textName + "Text");
    }
}
private String getTextMessage(String message) {
    // CMC - Added Method

    //-------------------------------------------------
    // If the localeMgr was specified, use it,
    // otherwise, return the english text
    // based on the message given
    //-------------------------------------------------
    if (localeMgr != null) {
        return localeMgr.getString("UtilitiesJCON",
            message + "Text");
    } 
    else if (message == "ProceedToExit") {
            return ("Proceed to exit the system?");
    } 
    else if (message == "SystemConfirmation") {
       return ("System Confirmation");
    } 
    else if (message == "CannotCloseMsg") {
       return ("Sorry, cannot close now because:");
    } 
    else {
       throw new IllegalArgumentException(message 
             + " is not a valid message");
    }
}

//--- Std
private static void print(String text) {
    System.out.println("SystemCloser" + text);
}

} // End class
