package uhr.core.tron;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

/**
* This class defines a configurable key value data structure 
* similar to Hashtable. Its outstanding characteristic is it can
* also automatically support tree like behavior, allowing it be 
* used as a param, tree, etc. We expect to add more configurable
* properties.
* <p>
* Keys must be Strings with no spaces allowed. All configurable 
* properties have sensible defaults. The configurable
* properties are: <pre> <code>
*    isNullValueAllowed()
*    isOrderedKeys()
*    getNestedKeySeparator()
* </pre> </code>
* This implementation is not thread-safe. We expect to need to
* add methods to the interface as needs arise. For example there
* is no getParent(), which hinders full use as a tree.
*
* @author Jack Harich
*/
public class PropMapStd implements PropMap {

//---------- Internal Fields -------------------------------------
protected Hashtable properties; // Set in initConfiguration()
protected Vector    keys;       // null if not orderedKeys

protected boolean isNullValueAllowed;
protected String  keySeparator; // null for none
protected boolean isOrderedKeys;

// For insertBefore() use
protected boolean isInsert;
protected String keyBefore;

protected static final Object NULL_VALUE = new NullValue();

//---------- Initialization --------------------------------------
/**
* A no arg constructor that leaves configuration defaults alone.
* The defaults are null values are not allowed, keys are not 
* ordered, and no key separator.
*/
public PropMapStd() {
    initConfiguration();
}
/**
* A constructor for configuration.
*
* @param isNullValueAllowed determines whether null values are allowed.
* @param isOrderedKeys      determines whether ordered keys are maintained.
* @param keySeparator       the key separator to use or null for none.
*/
public PropMapStd(boolean isNullValueAllowed,
                          boolean isOrderedKeys,
                          String keySeparator) {
                           
    this.isNullValueAllowed = isNullValueAllowed;
    this.isOrderedKeys = isOrderedKeys;
    this.keySeparator = keySeparator;
    initConfiguration();
}
//---------- PropMap Implementation ------------------------------

//----- Configuration -----
/**
* Returns true if null values are allowed, false if not. The
* default is false.
* @return  whether null values are allowed or not.
*/
public boolean isNullValueAllowed() {
    return isNullValueAllowed;
}
/**
* Returns true if ordered keys are maintained, false if not.
* The default is false.
* <p>
* If true the keys are maintained in the order added. This 
* allows the order of keys to be preserved, instead of being 
* random. An example of usefulness is to preserve the order of a
* visual tree or list. 
* @return  whether ordered keys are maintained or not.
*/
public boolean isOrderedKeys() {
    return isOrderedKeys;
}
/**
* Returns the nested key separator or null if none in use.
* The default is null.
* <p>
* The separator is used for reads and writes with nested
* keys, such as "State.County.Town". This allows very efficient 
* use of keys for tree structures. This is only useful if a tree
* is being used and single keys don't have the separator. The 
* default is null, meaning none. Common, safe key separators are
* "." and "/".
* <p>
* If a key separator has been set, reads and writes will use 
* automatic nesting, which occurs if a separator is present in the 
* key.
* <p>
* For example put("Georgia.Dekalb.Clarkston", "2000") would put
* "2000" into the path using the key. The call 
* get("Georgia.Dekalb.Clarkston") would return "2000", as expected.
* But the call get("Georgia.Dekalb") returns an OmniTron, and can
* be followed by omniTron.get("Clarkston") which returns "2000". 
* Thus nested OmniTrons can be used to create a clean, easy to use,
* tree structure.
* @return  the nested key separator or null if none.
*/
public String getNestedKeySeparator() {
    return keySeparator;
}
/**
* Returns true if a non-null nested key separator has been set. 
* The default is false.
* @return  whether a nested key separator is in use fot a tree
*          data structure or not.
*/
public boolean isTree() {
    return (keySeparator != null ? true : false);
}
//----- Mutation -----
/**
* Puts the value in the collection and associates it with the key
* for retrival. Any previous value is overwritten.
* @param key   the unique identifier for the value.
* @param value the Object to associate with the key.
* @return      the previous value with the key, or null if none or
*              the value was null.
*/
// RECURSIVE
public Object put(String key, Object value) {
    if (properties == null) print(".put() - You forgot to initConfiguration()");

    // Validate key
    if (key.indexOf(" ") > -1) throw new IllegalArgumentException(
        "Spaces are not allowed in keys.");
       
    // Support null values 
    value = wrapValue(value);
    
    // Normal case, easy implementation
    if (! isTree()) return putKey(key, value);
    
    // Tree case
    Object previousValue = null;
    int separatorIndex = key.indexOf(keySeparator);
    if (separatorIndex > -1) {
        // key is a path
        String firstKey = key.substring(0, separatorIndex);
        String remainderKey = key.substring(separatorIndex + 1);
        PropMap propMap = (PropMap)properties.get(firstKey);
        // Create propMap if necessary
        if (propMap == null) {
            propMap = (PropMap)this.clone();
            propMap.removeAll();
            putKey(firstKey, propMap);
        }
        // put remaining path into firstKey - RECURSE
        propMap.put(remainderKey, value);

    } else {
        // key is not a path. Recursion ends here.
        previousValue = putKey(key, value);
    }
    return previousValue;
}
// put() helper. Returns previous value.
protected Object putKey(String key, Object value) {
//print(".putKey() - key = " + key + ", value = " + value); 
    Object previousValue = properties.put(key, value);
    if (! isInsert) {
        // For normal put()
        if (previousValue == null && isOrderedKeys) {
            keys.addElement(key);
        }
    } else {
        // For insertBefore()
        int index = keys.indexOf(keyBefore);
        keys.insertElementAt(key, index);
    }
    return previousValue;
}
/**
* Inserts the key and value before the keyBefore. This is only 
* allowed for ordered collections. The keyBefore must already be
* in the collection and the key must not already be in the 
* collection.
* @param key        the unique identifier for the value.
* @param value      the Object to associate with the key.
* @param keyBefore  the key to insert the key/value before.
*/ 
public void insertBefore(String key, Object value, String keyBefore) {
    if (! isOrderedKeys) throw new IllegalStateException(
        "Cannot insertBefore since not ordered keys.");
        
    if (! containsKey(keyBefore)) throw new IllegalStateException(
        "Cannot insertBefore since keyBefore '" + keyBefore + "' is not found.");
        
    if (containsKey(key)) throw new IllegalStateException(
        "Cannot insertBefore since key '" + key + "' is already in use.");        
    
    isInsert = true;
        this.keyBefore = keyBefore;
        put(key, value);
    isInsert = false;
}
/**
* Removes the key and its value. 
* @param key  the key to remove.
* @return     the value removed, or null if none or the value was null.
*/
// RECURSIVE
public Object removeKey(String key) {
    // Case of key not in collection
    if (! properties.containsKey(key)) return null;
    
    // Normal case, easy implementation
    if (! isTree()) {
        if (isOrderedKeys) keys.removeElement(key);
        Object value = properties.remove(key);
        return unwrapValue(value);
    }
    // Tree case
    Object previousValue = null;
    int separatorIndex = key.indexOf(keySeparator);
    if (separatorIndex > -1) {
        // key is a path
        String firstKey = key.substring(0, separatorIndex);
        String remainderKey = key.substring(separatorIndex + 1);
        PropMap firstPropMap = (PropMap)properties.get(firstKey);
        // Remove key from firstKey - RECURSE
        firstPropMap.removeKey(remainderKey);
        // Remove firstPropMap if no more keys
        if (firstPropMap.isEmpty()) {
            properties.remove(firstKey);
        }
    } else {
        // key is not a path. Recursion ends here.
        previousValue = removeKeyHere(key);
    }
    return previousValue;
}
// removeKey() helper. Returns previous value.
protected Object removeKeyHere(String key) {
    Object previousValue = properties.remove(key);
    if (previousValue != null && isOrderedKeys) {
        keys.removeElement(key);
    } 
    return previousValue;
}
/**
* Removes all occurances of the key for this value. If the value is
* null and null values are allowed, then all keys with null values
* are removed, so be careful. :-)
* @param value  the value to remove all occurances of.
* @return       the number of keys removed.
*/
public int removeValue(Object value) {   
    value = wrapValue(value);
    int count = 0;
    Enumeration enumKeys = properties.keys();
    while (enumKeys.hasMoreElements()) {
        Object key = enumKeys.nextElement();
        Object foundValue = properties.get(key);
        if (value.equals(foundValue)) {
            properties.remove(key);
            if (isOrderedKeys) keys.removeElement(key);
            count++;
        }
    }
    return count;
}
/**
* Removes all keys and values from the collection. Has no effect
* on the configuration.
*/
public void removeAll() {
    // Note the inconsistency in method name here
    properties.clear();
    if (isOrderedKeys) keys.removeAllElements();       
}
//----- Readers -----
/**
* Returns the value to which the specified key is mapped in this
* collection. 
* @param key  the key of the value to retrieve.
* @return     the value for the key, or null if not found or null if
* the value was null.
*/
// RECURSIVE
public Object get(String key) {
    if (! isTree()) {
        return unwrapValue(properties.get(key));
    }
    // Tree
    int separatorIndex = key.indexOf(keySeparator);
    if (separatorIndex > -1) {
        // key is a path
        String firstKey = key.substring(0, separatorIndex);
        String remainderKey = key.substring(separatorIndex + 1);
        PropMap propMap = (PropMap)properties.get(firstKey);
        // Get key from nested propMap - RECURSE
        return propMap.get(remainderKey);
    } else {
        // key is not a path. Recursion ends here.
        return unwrapValue(properties.get(key));
    }
}
/**
* Returns an enumeration of the keys in this collection.  
* @return  an enumeration of all the keys, in order if ordered keys.
*/
public Enumeration getKeys() {
    if (isOrderedKeys) {
        return keys.elements();
    } else {
        return properties.keys();
    }
}
//----- General -----
/**
* Returns the number of key values in the collection, which may be
* zero. Note this doesn't include children if a tree.
* @return  the size of the collection.
*/
public int getSize() {
    return properties.size();
}
/**
* Returns true if the collection is empty or false if not.
* @return  true if empty or false if not.
*/
public boolean isEmpty() {
    return properties.isEmpty();
}
/**
* Returns true if the collection contains the key, false if not.
* @param key  the key to test.
* @return     true if the collection contains the key, false if not.
*/
public boolean containsKey(String key) {
    return properties.containsKey(key);
}
/**
* Returns true if the collection contains the value, false if not.
* Note that if null values are allowed and the value is null, then
* the value may be associated with more than one key.
* @param value  the value to test.
* @return       true if the collection contains the value, false if not.
*/
public boolean containsValue(Object value) {
    value = wrapValue(value);
    return properties.contains(value);
}
/**
* Returns the first key found for the value, or null if not found. 
* This may be only useful for ordered keys.
* @param value  the value to use to find the key.
* @return       the key found or null if not foubd.
*/
public String findFirstKey(Object value) {
    value = wrapValue(value);
    Enumeration enumKeys = getKeys();
    while (enumKeys.hasMoreElements()) {
        Object key = enumKeys.nextElement();
        Object foundValue = properties.get(key);
        if (value.equals(foundValue)) return (String)key;    
    }
    return null;
}
/**
* Returns a comma delimited string of key/values, for  example: 
* "[Name=Pantajeli, Age=11]". If a tree then a tree format is used.
* Both are designed to be easily readable.
* @return  the String representation of the collection.
*/
// Overly complex due to mixing normal and tree
public String toString() {
    boolean isTree = isTree();
    int level = 0;
    StringBuffer text = new StringBuffer();
    if (! isTree) text.append("[");
    Enumeration enumKeys = getKeys();
    
    while (enumKeys.hasMoreElements()) {
        String key = (String)enumKeys.nextElement();
        Object value = unwrapValue(properties.get(key));
        if (! isTree) {
            text.append(key + "=" + stringValue(value));
            if (enumKeys.hasMoreElements()) text.append(", ");        
        } else {
            // Tree
            if (value instanceof PropMap) {
                text.append(key + "\n");
                text.append(listTree((PropMap)value, level + 1));
                    
            } else {
                // Key is not a path
                text.append(key + "=" + stringValue(value));
            } 
            if (enumKeys.hasMoreElements()) text.append("\n");
        }      
    }   
    if (! isTree) text.append("]");
    return text.toString();
}
// toString() helper. RECURSIVE.
protected String listTree(PropMap propMap, int level) {
    StringBuffer text = new StringBuffer();
    Enumeration enumKeys = propMap.getKeys();
    
    while (enumKeys.hasMoreElements()) {
        String key = (String)enumKeys.nextElement();
        Object value = unwrapValue(propMap.get(key));
        // Indent using level
        // Add prefix of spaces
        for (int j = 1; j <= level; j++) {
            text.append("    "); // 4 spaces per level
        }
        if (value instanceof PropMap) {
            text.append(key + "\n");
            // RECURSE
            text.append(listTree((PropMap)value, level + 1));
                    
        } else {
            // Key is not a path. Recursion ends here.
            text.append(key + "=" + stringValue(value));
            if (enumKeys.hasMoreElements()) text.append("\n");
        }
    }   
    return text.toString();
}
// toString() helper
protected String stringValue(Object value) {
    return (value == null ? "null" : value.toString() );
}
/**
* Returns a shallow copy of the object.
* @return  a shallow copy of the object.
*/
public Object clone() {
    try {
        PropMapStd propMap = (PropMapStd)super.clone();
        propMap.properties = (Hashtable)properties.clone();
        if (keys != null) propMap.keys = (Vector)keys.clone();
        return propMap;
        
    } catch(CloneNotSupportedException ex) {
        print(",clone() - This should not happen. There is " +
            "probably a value that is uncloneable.");
        throw new RuntimeException("Clone failed.");
    }
}
//---------- ConvenientStringMap throuhg PropMap -----------------
// From Steve's PropMapStd, with changes

// ----- String methods
/**
* Sets the key value, overwriting any that already exists.
* @param key   the unique key, which may not be null.
* @param value the String value, which may be null.
*/
public void setString(String key, String value) {
    put(key, value);
}
/**
* Returns the String value for the key.
* @param key  the key to be used for the lookup.
* @return     the key's value or null.
*/
public String getString(String key) {
    Object value = get(key);
    return (value == null) ? null : value.toString();
}
/**
* Same as getString(key) except allows a default to be
* returned rather than null.
* @param key  the key to be used for the lookup.
* @param defaultValue the default to be used if the
*                     return would otherwise be null.
* @return     the key's value or the defaultValue if not found.
*/
public String getStringDefault(String key, String defaultValue) {
    Object value = get(key);
    return (value == null) ? defaultValue : value.toString();
}
// ----- boolean methods
/**
* Sets the key value, overwriting any that already exists.
* @param key   the unique key, which may not be null.
* @param value the boolean value.
*/
public void setBoolean(String key, boolean value) {
    put(key, value ? Boolean.TRUE : Boolean.FALSE);
}
/**
* Same as setBoolean(key, true).
*/
public void setTrue(String key) {
    put(key, Boolean.TRUE);
}
/**
* Same as setBoolean(key, false).
*/
public void setFalse(String key) {
    put(key, Boolean.FALSE);
}
/**
* Returns the boolean value for the key.
* We assume that the setBoolean method was used to set this value.
* If a non-Boolean value is found, an IllegalStateException is thrown.
* If no value is set, an IllegalStateException is thrown.
* @param key  the key to be used for the lookup.
* @return     the key's value or an exception if not found.
*/
public boolean getBoolean(String key) {
    // optimistic implementation :)
    try {
        return ((Boolean) get(key)).booleanValue();
    } catch (ClassCastException cce) {
        throw new IllegalStateException("Stored value was not a Boolean");
    } catch (NullPointerException npe) {
        if  (containsKey(key)) {
            throw new IllegalStateException("Stored value was null");
        } else {
            throw new IllegalStateException("Key "+key+" has no associated value");
        }
    }
}
/**
* Used to determine if the key value is true.
* @param key  the key in question.
* @return     true if the key's value is true, false if
*    it's false, or an exception if not found.
*/
public boolean isTrue(String key) {
    return getBoolean(key) == true;
}
/**
* Same as isTrue(key) except allows a default to be
* returned if not found rather than an exception.
* @param key  the key to be used for the lookup.
* @param defaultValue the default to be used if the
*    value is not found.
* @return     true if the key's value is true, false if
*    it's false, or the defaultValue if not found.
*/
public boolean isTrueDefault(String key, boolean defaultValue) {
    Object value = get(key);
    if (value != null && value instanceof Boolean) {
        return ((Boolean) value).booleanValue();
    } else {
        return defaultValue;
    }
}
/**
* Used to determine if the key value is false.
* @param key  the key in question.
* @return     true if the key's value is false, false if
*    it's true, or an exception if not found.
*/
public boolean isFalse(String key) {
    return ! isTrue(key);
}
/**
* Same as isFalse(key) except allows a default to be
* returned if not found rather than an exception.
* @param key  the key to be used for the lookup.
* @param defaultValue the default to be used if the
*    value is not found.
* @return     true if the key's value is false, false if
*    it's true, or the defaultValue if not found.
*/
public boolean isFalseDefault(String key, boolean defaultValue) {
// watch out for this... it is a bit tricky!
    return ! isTrueDefault(key, ! defaultValue);
}
// ----- int methods
/**
* Sets the key value, overwriting any that already exists.
* @param key   the unique key, which may not be null.
* @param value the int value.
*/
public void setInt(String key, int value) {
    put(key, new Integer(value));
}
/**
* Returns the int value for the key.
* @param key  the key to be used for the lookup.
* @return     the key's value or an exception if not found.
*/
public int getInt(String key) {
    // optimistic implementation :)
    try {
        return ((Integer) get(key)).intValue();
    } catch (Exception ex) {
        handleTypeException(ex, key, "int");
        return 0;
    }        
}
/**
* Same as getInt(key) except allows a default to be
* returned rather than an exception if not found.
* @param key  the key to be used for the lookup.
* @param defaultValue the default to be used if not found.
* @return the key's value or the defaultValue if not found.
*/
public int getIntDefault(String key, int defaultValue) {
    Object value = get(key);
    if (value != null && value instanceof Integer) {
        return ((Integer) value).intValue();
    } else {
        return defaultValue;
    }
}
// ----- long methods
/**
* Sets the key value, overwriting any that already exists.
* @param key   the unique key, which may not be null.
* @param value the long value.
*/
public void setLong(String key, long value) {
    put(key, new Long(value));
}
/**
* Returns the long value for the key.
* @param key  the key to be used for the lookup.
* @return     the key's value or an exception if not found.
*/
public long getLong(String key) {
    // optimistic implementation :)
    try {
        return ((Long) get(key)).longValue();
    } catch (Exception ex) {
        handleTypeException(ex, key, "long");
        return 0;
    }        
}
/**
* Same as getLong(key) except allows a default to be
* returned rather than an exception if not found.
* @param key  the key to be used for the lookup.
* @param defaultValue the default to be used if not found.
* @return the key's value or the defaultValue if not found.
*/
public long getLongDefault(String key, long defaultValue) {
    Object value = get(key);
    if (value != null && value instanceof Long) {
        return ((Long) value).longValue();
    } else {
        return defaultValue;
    }
}
// ----- float methods
/**
* Sets the key value, overwriting any that already exists.
* @param key   the unique key, which may not be null.
* @param value the float value.
*/
public void setFloat(String key, float value) {
    put(key, new Float(value));
}
/**
* Returns the float value for the key.
* @param key  the key to be used for the lookup.
* @return     the key's value or an exception if not found.
*/
public float getFloat(String key) {
    // optimistic implementation :)
    try {
        return ((Float) get(key)).floatValue();
    } catch (Exception ex) {
        handleTypeException(ex, key, "float");
        return 0;
    }        
}
/**
* Same as getFloat(key) except allows a default to be
* returned rather than an exception if not found.
* @param key  the key to be used for the lookup.
* @param defaultValue the default to be used if not found.
* @return the key's value or the defaultValue if not found.
*/
public float getFloatDefault(String key, float defaultValue) {
    Object value = get(key);
    if (value != null && value instanceof Float) {
        return ((Float) value).floatValue();
    } else {
        return defaultValue;
    }
}
// ----- double methods
/**
* Sets the key value, overwriting any that already exists.
* @param key   the unique key, which may not be null.
* @param value the double value.
*/
public void setDouble(String key, double value) {
    put(key, new Double(value));
}
/**
* Returns the double value for the key.
* @param key  the key to be used for the lookup.
* @return     the key's value or an exception if not found.
*/
public double getDouble(String key) {
    // optimistic implementation :)
    try {
        return ((Double)get(key)).doubleValue();
    } catch (Exception ex) {
        handleTypeException(ex, key, "double");
        return 0;
    }
}
/**
* Same as getDouble(key) except allows a default to be
* returned rather than an exception if not found.
* @param key  the key to be used for the lookup.
* @param defaultValue the default to be used if not found.
* @return the key's value or the defaultValue if not found.
*/
public double getDoubleDefault(String key, double defaultValue) {
    Object value = get(key);
    if (value != null && value instanceof Double) {
        return ((Double) value).doubleValue();
    } else {
        return defaultValue;
    }
}
//---------- Protected Methods -----------------------------------
/**
* This must be called before collection use. 
*/
protected void initConfiguration() {
    properties = new Hashtable();
    if (isOrderedKeys) keys = new Vector();
}
/**
* Prepares value for storage, supporting nulls if allowed
* Wrap and unwrap are similar to Steve's encapsulateValue(), etc.
*/
protected Object wrapValue(Object value) {
    if (value == null && ! isNullValueAllowed) throw new
        IllegalArgumentException("Null values are not allowed.");
        
    return (value == null ? NULL_VALUE : value);
}
/**
* Unwrap value from storage, returning original, which may be null
*/
protected Object unwrapValue(Object value) {
    return (NULL_VALUE.equals(value) ? null : value);
}
/**
* For handline conversion exceptions.
*/
protected void handleTypeException(Exception ex, String key, String type) {
    if (ex instanceof ClassCastException) {
        throw new IllegalStateException("Stored value was not a " + type);
      
    } else if (ex instanceof NullPointerException) {
        if  (containsKey(key)) {
            throw new IllegalStateException("Stored value was null");
        } else {
            throw new IllegalStateException("Key "+key+" has no associated value");
        }
    } else {
        throw new RuntimeException(ex.getMessage() +
            "\nProblem with key '" + key + "' of type " + type + ".");
    }
}
//---------- Standard --------------------------------------------
private static void print(String text) {
    System.out.println("PropMapStd" + text);
}
//---------- Inner Classes ---------------------------------------
/**
* From Steve's earlier implementation of PropMapStd. Very nice.
*/
protected static final class NullValue {
    // All NullValue instances are equal to each other
    public boolean equals(Object obj) {
        return ((obj != null && obj instanceof NullValue) ? true : false);
    }
    // This should never get printed to the screen.
    public String toString() {
        return "** instance of NullValue **";
    }     
}


} // End class