package uhr.core.tron;

import java.util.Enumeration;

/**
* This interface defines a configurable key value data structure 
* similar to Hashtable. Its outstanding characteristic is it can
* also automatically support tree like behavior, allowing it be 
* used as a param, tree, etc. We expect to add more configurable
* properties.
* <p>
* Keys must be Strings with no spaces allowed. All configurable 
* properties have sensible defaults. The configurable
* properties are: <pre> <code>
*    isNullValueAllowed()
*    isOrderedKeys()
*    getNestedKeySeparator()
* </pre> </code>
* The implementation should have a no arg constructor and other
* constructors for configuration.
*
* @author Jack Harich 
*/
public interface PropMap extends ConvenientStringMap, Cloneable,
                                 java.io.Serializable {

//----- Configuration -----
/**
* Returns true if null values are allowed, false if not. The
* default is false.
* @return  whether null values are allowed or not.
*/
public boolean isNullValueAllowed();

/**
* Returns true if ordered keys are maintained, false if not.
* The default is false.
* <p>
* If true the keys are maintained in the order added. This 
* allows the order of keys to be preserved, instead of being 
* random. An example of usefulness is to preserve the order of a
* visual tree or list. 
* @return  whether ordered keys are maintained or not.
*/
public boolean isOrderedKeys();

/**
* Returns the nested key separator or null if none in use.
* The default is null.
* <p>
* The separator is used for reads and writes with nested
* keys, such as "State.County.Town". This allows very efficient 
* use of keys for tree structures. This is only useful if a tree
* is being used and single keys don't have the separator. The 
* default is null, meaning none. Common, safe key separators are
* "." and "/".
* <p>
* If a key separator has been set, reads and writes will use 
* automatic nesting, which occurs if a separator is present in the 
* key.
* <p>
* For example put("Georgia.Dekalb.Clarkston", "2000") would put
* "2000" into the path using the key. The call 
* get("Georgia.Dekalb.Clarkston") would return "2000", as expected.
* But the call get("Georgia.Dekalb") returns an OmniTron, and can
* be followed by omniTron.get("Clarkston") which returns "2000". 
* Thus nested OmniTrons can be used to create a clean, easy to use,
* tree structure.
* @return  the nested key separator or null if none.
*/
public String getNestedKeySeparator(); 

/**
* Returns true if a non-null nested key separator has been set. 
* The default is false.
* @return  whether a nested key separator is in use fot a tree
*          data structure or not.
*/
public boolean isTree();

//----- Mutation -----
/**
* Puts the value in the collection and associates it with the key
* for retrival. Any previous value is overwritten.
* @param key   the unique identifier for the value.
* @param value the Object to associate with the key.
* @return      the previous value with the key, or null if none or
*              the value was null.
*/
public Object put(String key, Object value);

/**
* Inserts the key and value before the keyBefore. This is only 
* allowed for ordered collections. The keyBefore must already be
* in the collection and the key must not already be in the 
* collection.
* @param key        the unique identifier for the value.
* @param value      the Object to associate with the key.
* @param keyBefore  the key to insert the key/value before.
*/ 
public void insertBefore(String key, Object value, String keyBefore);

/**
* Removes the key and its value. 
* @param key  the key to remove.
* @return     the value removed, or null if none or the value was null.
*/
public Object removeKey(String key);

/**
* Removes all occurances of the key for this value. If the value is
* null and null values are allowed, then all keys with null values
* are removed, so be careful. :-)
* @param value  the value to remove all occurances of.
* @return       the number of keys removed.
*/
public int removeValue(Object value);

/**
* Removes all keys and values from the collection. Has no effect
* on the configuration.
*/
public void removeAll();

//----- Readers -----
/**
* Returns the value to which the specified key is mapped in this
* collection. 
* @param key  the key of the value to retrieve.
* @return     the value for the key, or null if not found or null if
* the value was null.
*/
public Object get(String key);

/**
* Returns an enumeration of the keys in this collection.  
* @return  an enumeration of all the keys, in order if ordered keys.
*/
public Enumeration getKeys();

//public Enumeration getValues(); // later

//----- General -----
/**
* Returns the number of key values in the collection, which may be
* zero. Note this doesn't include children if a tree.
* @return  the size of the collection.
*/
public int getSize();

/**
* Returns true if the collection is empty or false if not.
* @return  true if empty or false if not.
*/
public boolean isEmpty();

/**
* Returns true if the collection contains the key, false if not.
* @param key  the key to test.
* @return     true if the collection contains the key, false if not.
*/
public boolean containsKey(String key);

/**
* Returns true if the collection contains the value, false if not.
* Note that if null values are allowed and the value is null, then
* the value may be associated with more than one key.
* @param value  the value to test.
* @return       true if the collection contains the value, false if not.
*/
public boolean containsValue(Object value);


/**
* Returns the first key found for the value, or null if not found. 
* This may be only useful for ordered keys.
* @param value  the value to use to find the key.
* @return       the key found or null if not foubd.
*/
public String findFirstKey(Object value);

/**
* Returns a comma delimited string of key/values, for  example: 
* "[Name=Pantajeli, Age=11]". If a tree then a tree format is used.
* Both are designed to be easily readable.
* @return  the String representation of the collection.
*/
public String toString();

/**
* Returns a shallow copy of the object.
* @return  a shallow copy of the object.
*/
public Object clone();


} // End interface