package uhr.core.tron;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

/**
* This interface defines a flexible datatron that can meet a vast
* variety of needs. See
* <a href="uhr.core.tron.Flexitron.html">full documentation</a>.
* <p>
* Implementation status - No key path, which is terribly complex.
*
* @author Jack Harich
*/
public class FlexitronStd extends ConvenientStringMapStd
                                             implements Flexitron {

//---------- Internal Fields -------------------------------------
//----- Configurable properties, set only by constuctor
protected boolean isElementalOnly;
protected boolean isAutoDetectDuplicateKeys;
protected boolean isNullValueAllowed;
protected String  keySeparator; // null if none

//----- Internal use
protected boolean isTree; // true if Flexitron added

// Key = key, Object = value
// hash is null if isElemental()
protected Hashtable hash;

// Contains Elements in order added
protected Vector elements = new Vector();

protected static final Object NULL_VALUE = new NullValue();

//---------- Initialization --------------------------------------
/**
* Creates a new FlexitronStd with the default configuration. This is
* the same as <code>new FlexitronStd(false, false, false, null)</code>.
*/
public FlexitronStd() {
    this(false, false, false, null);
}
/**
* Creates a new FlexitronStd configured with the values provided.
* <p>
* For use as a data structure for parsed XML, we recommend: <pre>
*   - isElementalOnly = false
*   - isAutoDetectDuplicateKeys = true
*   - isNullValueAllowed = true (empty tags are null)
*   - keySeparator = "/"
* </pre>
* @param isElementalOnly  true to maintain elements only, false
*   to maintain elements and keyed values.
*
* @param isAutoDetectDuplicateKeys  true to automatically switch
*   to isElemental() if a duplicate key is added, false to not
*   allow duplicate keys.
*
* @param isNullValueAllowed  true to allow null values, false to 
*   not allow null values.
*
* @param keySeparator  the key separator to use or null for none.
*/
public FlexitronStd(boolean isElementalOnly,
                    boolean isAutoDetectDuplicateKeys,
                    boolean isNullValueAllowed,
                    String  keySeparator) {
                    
    this.isElementalOnly = isElementalOnly;
    this.isAutoDetectDuplicateKeys = isAutoDetectDuplicateKeys;
    this.isNullValueAllowed = isNullValueAllowed;
    this.keySeparator = keySeparator;
    initHash();
}
protected void initHash() {
    if (! isElementalOnly) hash = new Hashtable();;
}
//---------- ConvenientStringMapStd Abstract Methods -------------
/**
* The setter for the subclass to implement. This may be a put or
* add or such.
*
* @param key  the key, which may not be null.
* @param value  the value, which may be null if allowed.
*/
protected void setSubclass(String key, Object value) {
    add(key, value);
}
/**
* The getter for the subclass to implement.
*
* @param key  the key, which may not be null.
* @return  the value for the key or null if none. If null
*    values are allowed the null may be returned as the real value.
*/
protected Object getSubclass(String key) {
    return get(key);
}
/**
* Determines whether the collection contains the key.
* @return  true if the collection contains the key, false if not.
*/
protected boolean containsKeySubclass(String key) {
    return containsKey(key);
}
//---------- Public Methods --------------------------------------
//----- Configuration Readers
/**
* Determines whether if only elements are maintained. The default
* is false. True is the same as the collection behaving like a 
* Vector. False allows maintaining keyed values and the
* elements.
*
* @return true if only elements are maintained, false if not.
*/
public boolean isElementalOnly() {
    return isElementalOnly;
} 
/**
* Determines whether the collection will automatically switch to
* isElemental() if a duplicate key is added. True allows this to
* happen, false prevents it and duplicate keys.
*
* @return true if the feature is on, false if off.
*/
public boolean isAutoDetectDuplicateKeys() {
    return isAutoDetectDuplicateKeys;
}
/**
* Determines whether null values are allowed. A null can be a
* perfectly valid data value in some designs, so we support it.
*
* @return true if null values are allowed, false if not.
*/
public boolean isNullValueAllowed() {
    return isNullValueAllowed;
}
/**
* Returns the nested key separator or null if none in use.
* The default is null.
* <p>
* The separator is used for reads with nested keys, such as
* get("State/County/Georgia.Fulton") where "/" is the separator.
* We recommend not using "." as the separator to allow the use of
* more meaningful keys. Naturally we recommend "/" for the 
* separator.
*/
public String getKeySeparator() {
    return keySeparator;
}
/**
* Determines whether the collection has Elements or key values.
* @return  true if is using Elements, false if using key values.
*/
public boolean isElemental() {
    return (hash == null ? true : false);
}
//----- Mutators
/**
* Adds the value in the collection and associates it with the key
* for retrival. If the key is a duplicate and IsAutoDetectDuplicateKeys
* is true, then the key and value are put into an Element, and
* the collection is said to be "elemental".
*
* @param key   the unique identifier for the value.
* @param value the Object to associate with the key.
* @exception IllegalStateException  if the key is a duplicate and
*   IsAutoDetectDuplicateKeys is false.
* @exception IllegalArgumentException if the key is null.
*/
public void add(String key, Object value) {
    if (key == null) throw new IllegalArgumentException(
        "The key cannot be null.");
    if (hash != null) {
        // Support null values 
        value = wrapValue(value);
    
        Object previousValue = hash.put(key, value);
        if (previousValue != null) {
            if (! isAutoDetectDuplicateKeys) {
                throw new IllegalStateException("Cannot add key '" +
                key + "' because it's a duplicate and isAutoDetectDuplicateKeys = false.");
            } else {
                hash = null; // Stop hashing
            }
        }
    }
    // Note the value is not wrapped
    elements.addElement(new Element(key, value));
    if (value instanceof Flexitron) isTree = true;
}
/**
* Removes the key and its value. This is only for not is Elemental.
* 
* @param key  the key to remove.
* @return     the value removed, or null if none or the value was null.
* @exception IllegalStateException  if the collection is Elemental.
*/
public Object removeKey(String key) {
    if (hash == null) throw new IllegalStateException(
        "Cannot remove key '" + key + "' since isElemental().");
    Object value = hash.remove(key);
    if (value != null) {
        elements.removeElement(new Element(key, value));    
    }
    return unwrapValue(value);
}
/**
* Removes the <b>first occurance</b> of the element from the collection.
* Since you may not have the actual element to remove, create a
* new one with the desired key and value. This is used to match
* the one to remove. This is only for is Elemental.
*
* @param element  the element to remove.
* @return  true if the element was removed, false if not.
* @exception IllegalStateException  if the collection is not Elemental.
*/
public boolean removeElement(Element element) {
    if (hash != null) throw new IllegalStateException(
        "Cannot remove element since the collection is not elemental.");
    return elements.removeElement(element);
}
/**
* Removes all keys and values from the collection. Has no effect
* on the configuration.
*/
public void removeAll() {
    if (hash == null) {
        initHash();
        // was ***hash = new Hashtable();
    } else {
        hash.clear(); // 99
    }
    elements.removeAllElements();
}
//----- Basic Readers
/**
* Returns the value to which the specified key is mapped in this
* collection. Do not use if isElemental().
* <p>
* If a key separator is in use and the separator is present in the
* key, then the nested key value is returned. Note this assumes
* that nested keys except the last are Flexitrons. For example
* the key "State/County/Georgia.Fulton" implies that a Flexitron
* was added to this collection with the key of "State" and it
* contained a key of "County" with a Flexitron value that had a
* key of "Georgia.Fulton" with an arbitrary value, which is what
* is returned.
*
* @param key  the key of the value to retrieve.
* @return     the value for the key, or null if not found or null
*             if the value was null.
* @exception IllegalStateException   if isElemental() or if the
* nested key is incompatable with the actual contents.
*/
public Object get(String key) {
    if (hash == null) throw new IllegalStateException(
        "Cannot get key '" + key + "' because collection is elemental.");
    
    if (! isTree) {
        return unwrapValue(hash.get(key));
    }
    // Tree
    int separatorIndex = key.indexOf(keySeparator);
    if (separatorIndex > -1) {
        // key is a path
        String firstKey = key.substring(0, separatorIndex);
        String remainderKey = key.substring(separatorIndex + 1);
        Flexitron tron = (Flexitron)hash.get(firstKey);
        // Get key from nested Flexitron - RECURSE
        return tron.get(remainderKey);
    } else {
        // key is not a path. Recursion ends here.
        return unwrapValue(hash.get(key));
    }
}
/**
* Returns an enumeration of the keys in this collection.  
* Do not use if isElemental().
*
* @return  an enumeration of all the keys, in order if ordered keys.
* @exception IllegalStateException   if isElemental();
*/
public Enumeration getKeys() {
    if (hash == null) throw new IllegalStateException(
        "Cannot getKeys() because collection is elemental.");
    return hash.keys();
}
/**
* Returns an enumeration of the elements in this collection.  
*
* @return  an enumeration of all the elements in order.
*/
public Enumeration getElements() {
    return elements.elements();
}
//----- General
/**
* Returns the number of key values or elements in the collection,
* which may be zero. Note this doesn't include children if a tree.
* @return  the size of the collection.
*/
public int getSize() {
    return elements.size();
}
/**
* Returns true if the collection is empty or false if not.
* @return  true if empty or false if not.
*/
public boolean isEmpty() {
    return elements.isEmpty();
}
/**
* Returns true if the collection contains the key, false if not.
* Do not use if isElemental().
*
* @param key  the key to test.
* @return     true if the collection contains the key, false if not.
* @exception IllegalStateException   if isElemental();
*/
public boolean containsKey(String key) {
    if (hash == null) throw new IllegalStateException(
        "Cannot do containsKey() for key '" + key + "' because collection is elemental.");
    return hash.containsKey(key);
}
/**
* Returns true if the collection contains the element, false if not.
*
* @param element  the element to test.
* @return     true if the collection contains the element, false if not.
*/
public boolean containsElement(Element element) {
    return elements.contains(element);
}
/**
* Returns a comma delimited string of key/values, for  example: 
* "[Name=Pantajeli, Age=11]". If a tree then a tree format is used.
* Both are designed to be easily readable.
* @return  the String representation of the collection.
*/
// Overly complex due to mixing normal and tree
public String toString() {
    int level = 0;
    StringBuffer text = new StringBuffer();
    if (! isTree) text.append("[");
    Enumeration enum = elements.elements();
    
    while (enum.hasMoreElements()) {
        Element element = (Element)enum.nextElement();
        String key = element.getKey();
        Object value = unwrapValue(element.getValue());
        if (! isTree) {
            text.append(key + "=" + stringValue(value));
            if (enum.hasMoreElements()) text.append(", ");        
        } else {
            // Tree
            if (value instanceof Flexitron) {
                text.append(key + "\n");
                text.append(listTree((Flexitron)value, level + 1));
                    
            } else {
                // Key is not a path
                text.append(key + "=" + stringValue(value));
            } 
            if (enum.hasMoreElements()) text.append("\n");
        }      
    }   
    if (! isTree) text.append("]");
    return text.toString();
}
//---------- Protected Methods ------------------------------------
// toString() helper. RECURSIVE.
protected String listTree(Flexitron tron, int level) {

    StringBuffer text = new StringBuffer();
    Enumeration enum = tron.getElements();
    
    while (enum.hasMoreElements()) {
        Element element = (Element)enum.nextElement();
        String key = element.getKey();
        Object value = unwrapValue(element.getValue());
        // Indent using level
        // Add prefix of spaces
        for (int j = 1; j <= level; j++) {
            text.append("    "); // 4 spaces per level
        }
        if (value instanceof Flexitron) {
            text.append(key + "\n");
            // RECURSE
            text.append(listTree((Flexitron)value, level + 1));
                    
        } else {
            // Key is not a path. Recursion ends here.
            text.append(key + "=" + stringValue(value));
        }
        if (enum.hasMoreElements()) text.append("\n");
    }   
    return text.toString();
}
// toString() helper
protected String stringValue(Object value) {
    return (value == null ? "null" : value.toString() );
}
/**
* Prepares value for storage, supporting nulls if allowed
* Wrap and unwrap are similar to Steve's encapsulateValue(), etc.
*/
protected Object wrapValue(Object value) {
    if (value == null && ! isNullValueAllowed) throw new
        IllegalArgumentException("Null values are not allowed.");
        
    return (value == null ? NULL_VALUE : value);
}
/**
* Unwrap value from storage, returning original, which may be null
*/
protected Object unwrapValue(Object value) {
    return (NULL_VALUE.equals(value) ? null : value);
}

//---------- Standard --------------------------------------------
private static void print(String text) {
    System.out.println("FlexitronStd" + text);
}
//---------- Inner Classes ---------------------------------------
/**
* From Steve's earlier implementation of PropMapStd. Very nice.
*/
protected static final class NullValue {
    // All NullValue instances are equal to each other
    public boolean equals(Object obj) {
        return ((obj != null && obj instanceof NullValue) ? true : false);
    }
    // This should never get printed to the screen.
    public String toString() {
        return "** instance of NullValue **";
    }     
}


} // End outer class