package uhr.core.tron;

import java.util.Enumeration;

/**
* This interface defines a flexible datatron that can meet a vast
* variety of needs. It's designed to be configured by a wrapper
* interface so that clients can merely use <code>new Wrapper()</code>.
* It's intended use is for receiving a parsed XML document, and
* for general all purpose use. We hope it will be used for many
* inter-part data structure needs, giving us all consistency and
* reuse.
* <p>
* By nesting Flexitrons one can build a tree datastructure.
* <p>
* The main features are:
* <BLOCKQUOTE>
* Keyed and Elements, or just Elements. If keyed then the usual
* collection of key values is maintained. If Elements then a 
* collection of Elements
* is maintained. Each Element has a key and value.
* <p>
* Add, remove key values or Elements. 
* <p>
* Get key values or enumerate Elements. It makes no sense to get
* an Element by key due to duplicate keys.
* <p>
* Automatic detection of duplicate keys. When a duplicate key
* is added, the Flexitron automatically switches from key values
* and Elements to just Elements. This simplifies parsing a 
* document into a data structure.
* <p>
* Key paths may be used to get values if Keyed. The key separator
* is configurable.
* <p>
* Many convenience methods for dealing with primitives.
* <p>
* Configurable for a variety of uses.
* </BLOCKQUOTE>
* Configuration is done in the implementation constructor. This
* prevents the client from fiddling with configuration, and makes
* the implementaiton easier because it can assume no further
* configuration is allowed once created. Wrappers extend Flexitron
* and the Wrapper constructor configures the Flexitron. The
* configurable properties are: 
* <BLOCKQUOTE>
* IsElementalOnly - Default is false. True to maintain elements
* only, false to maintain elements and keyed values.
* <p>
* IsAutoDetectDuplicateKeys - Default is false, 
* which means they are not allowed.
* <p>
* IsNullValuesAllowed - Default if false.
* <p>
* KeySeparator - Default is null, no tree.
* </BLOCKQUOTE>
* The default configuration is just like a Hashtable except that
* we use add instead of put, and overwrites are not allowed. To
* do an overwrite, first remove the existing key.
*
* @author Jack Harich
*/
public interface Flexitron extends ConvenientStringMap {

//---------- Public Methods --------------------------------------
//----- Configuration Readers
/**
* Determines whether if only elements are maintained. The default
* is false. True is the same as the collection behaving like a 
* Vector. False allows maintaining keyed values and the
* elements.
*
* @return true if only elements are maintained, false if not.
*/
public boolean isElementalOnly();
 
/**
* Determines whether the collection will automatically switch to
* isElemental() if a duplicate key is added. True allows this to
* happen, false prevents it and duplicate keys.
*
* @return true if the feature is on, false if off.
*/
public boolean isAutoDetectDuplicateKeys();

/**
* Determines whether null values are allowed. A null can be a
* perfectly valid data value in some designs, so we support it.
*
* @return true if null values are allowed, false if not.
*/
public boolean isNullValueAllowed();

/**
* Returns the nested key separator or null if none in use.
* The default is null.
* <p>
* The separator is used for reads with nested keys, such as
* get("State/County/Georgia.Fulton") where "/" is the separator.
* We recommend not using "." as the separator to allow the use of
* more meaningful keys. Naturally we recommend "/" for the 
* separator.
*/
public String getKeySeparator();

/**
* Determines whether the collection has Elements or key values.
* @return  true if is using Elements, false if using key values.
*/
public boolean isElemental();

//----- Mutators
/**
* Adds the value in the collection and associates it with the key
* for retrival. If the key is a duplicate and IsAutoDetectDuplicateKeys
* is true, then the key and value are put into an Element, and
* the collection is said to be "elemental".
* <p>
* If the value is a Flexitron then its values can be retreived
* with a nested key such as "State/County/Georgia.Fulton".
*
* @param key   the unique identifier for the value.
* @param value the Object to associate with the key.
* @exception IllegalStateException  if the key is a duplicate and
*   IsAutoDetectDuplicateKeys is false.
* @exception IllegalArgumentException if the key is null.
*/
public void add(String key, Object value);

/**
* Removes the key and its value. This is only for not is Elemental.
* 
* @param key  the key to remove.
* @return     the value removed, or null if none or the value was null.
* @exception IllegalStateException  if the collection is Elemental.
*/
public Object removeKey(String key);

/**
* Removes the <b>first occurance</b> of the element from the collection.
* Since you may not have the actual element to remove, create a
* new one with the desired key and value. This is used to match
* the one to remove. This is only for is Elemental.
*
* @param element  the element to remove.
* @return  true if the element was removed, false if not.
* @exception IllegalStateException  if the collection is not Elemental.
*/
public boolean removeElement(Element element);

/**
* Removes all keys and values from the collection. Has no effect
* on the configuration.
*/
public void removeAll();

//----- Basic Readers
/**
* Returns the value to which the specified key is mapped in this
* collection. Do not use if isElemental().
* <p>
* If a key separator is in use and the separator is present in the
* key, then the nested key value is returned. Note this assumes
* that nested keys except the last are Flexitrons. For example
* the key "State/County/Georgia.Fulton" implies that a Flexitron
* was added to this collection with the key of "State" and it
* contained a key of "County" with a Flexitron value that had a
* key of "Georgia.Fulton" with an arbitrary value, which is what
* is returned.
*
* @param key  the key of the value to retrieve.
* @return     the value for the key, or null if not found or null
*             if the value was null.
* @exception IllegalStateException   if isElemental() or if the
* nested key is incompatable with the actual contents.
*/
public Object get(String key);

/**
* Returns an enumeration of the keys in this collection.  
* Do not use if isElemental().
*
* @return  an enumeration of all the keys, in order if ordered keys.
* @exception IllegalStateException   if isElemental();
*/
public Enumeration getKeys();

/**
* Returns an enumeration of the elements in this collection.  
*
* @return  an enumeration of all the elements in order.
*/
public Enumeration getElements();

//----- General
/**
* Returns the number of key values or elements in the collection,
* which may be zero. Note this doesn't include children if a tree.
* @return  the size of the collection.
*/
public int getSize();

/**
* Returns true if the collection is empty or false if not.
* @return  true if empty or false if not.
*/
public boolean isEmpty();

/**
* Returns true if the collection contains the key, false if not.
* Do not use if isElemental().
*
* @param key  the key to test.
* @return     true if the collection contains the key, false if not.
* @exception IllegalStateException   if isElemental();
*/
public boolean containsKey(String key);

/**
* Returns true if the collection contains the element, false if not.
*
* @param element  the element to test.
* @return     true if the collection contains the element, false if not.
*/
public boolean containsElement(Element element);

/**
* Returns a comma delimited string of key/values, for  example: 
* "[Name=Pantajeli, Age=11]". If a tree then a tree format is used.
* Both are designed to be easily readable.
* @return  the String representation of the collection.
*/
// Overly complex due to mixing normal and tree
public String toString();

} // End interface