package uhr.core.tron;

/**
* This is an abstract class implementing ConvenientStringMap
* for data structure classes to subclass. The use of the
* Template Method pattern allows the subclasses to offer the full
* ease-of-use features ConvenientStringMap offers without the
* drudgery, bulk and lower reuse of an implementation.
*
* @author Jack Harich
*/
public abstract class ConvenientStringMapStd 
                            implements ConvenientStringMap {
                            
//---------- Abstract Methods ------------------------------------
/**
* The setter for the subclass to implement. This may be a put or
* add or such.
*
* @param key  the key, which may not be null.
* @param value  the value, which may be null if allowed.
*/
protected abstract void setSubclass(String key, Object value);

/**
* The getter for the subclass to implement.
*
* @param key  the key, which may not be null.
* @return  the value for the key or null if none. If null
*    values are allowed the null may be returned as the real value.
*/
protected abstract Object getSubclass(String key); 

/**
* Determines whether the collection contains the key.
* @return  true if the collection contains the key, false if not.
*/
protected abstract boolean containsKeySubclass(String key);

//---------- ConvenientStringMap Implementation ------------------
// From Steve's PropMapStd, with changes

// ----- String methods
/**
* Sets the key value, overwriting any that already exists.
* @param key   the unique key, which may not be null.
* @param value the String value, which may be null.
*/
public void setString(String key, String value) {
    setSubclass(key, value);
}
/**
* Returns the String value for the key.
* @param key  the key to be used for the lookup.
* @return     the key's value or null.
*/
public String getString(String key) {
    Object value = getSubclass(key);
    return (value == null) ? null : value.toString();
}
/**
* Same as getString(key) except allows a default to be
* returned rather than null.
* @param key  the key to be used for the lookup.
* @param defaultValue the default to be used if the
*                     return would otherwise be null.
* @return     the key's value or the defaultValue if not found.
*/
public String getStringDefault(String key, String defaultValue) {
    Object value = getSubclass(key);
    return (value == null) ? defaultValue : value.toString();
}
// ----- boolean methods
/**
* Sets the key value, overwriting any that already exists.
* @param key   the unique key, which may not be null.
* @param value the boolean value.
*/
public void setBoolean(String key, boolean value) {
    setSubclass(key, value ? Boolean.TRUE : Boolean.FALSE);
}
/**
* Same as setBoolean(key, true).
*/
public void setTrue(String key) {
    setSubclass(key, Boolean.TRUE);
}
/**
* Same as setBoolean(key, false).
*/
public void setFalse(String key) {
    setSubclass(key, Boolean.FALSE);
}
/**
* Returns the boolean value for the key.
* We assume that the setBoolean method was used to set this value.
* If a non-Boolean value is found, an IllegalStateException is thrown.
* If no value is set, an IllegalStateException is thrown.
* @param key  the key to be used for the lookup.
* @return     the key's value or an exception if not found.
*/
public boolean getBoolean(String key) {
    // optimistic implementation :)
    try {
        return ((Boolean) getSubclass(key)).booleanValue();
    } catch (ClassCastException cce) {
        throw new IllegalStateException("Stored value was not a Boolean");
    } catch (NullPointerException npe) {
        if  (containsKeySubclass(key)) {
            throw new IllegalStateException("Stored value was null");
        } else {
            throw new IllegalStateException("Key "+key+" has no associated value");
        }
    }
}
/**
* Used to determine if the key value is true.
* @param key  the key in question.
* @return     true if the key's value is true, false if
*    it's false, or an exception if not found.
*/
public boolean isTrue(String key) {
    return getBoolean(key) == true;
}
/**
* Same as isTrue(key) except allows a default to be
* returned if not found rather than an exception.
* @param key  the key to be used for the lookup.
* @param defaultValue the default to be used if the
*    value is not found.
* @return     true if the key's value is true, false if
*    it's false, or the defaultValue if not found.
*/
public boolean isTrueDefault(String key, boolean defaultValue) {
    Object value = getSubclass(key);
    if (value != null && value instanceof Boolean) {
        return ((Boolean) value).booleanValue();
    } else {
        return defaultValue;
    }
}
/**
* Used to determine if the key value is false.
* @param key  the key in question.
* @return     true if the key's value is false, false if
*    it's true, or an exception if not found.
*/
public boolean isFalse(String key) {
    return ! isTrue(key);
}
/**
* Same as isFalse(key) except allows a default to be
* returned if not found rather than an exception.
* @param key  the key to be used for the lookup.
* @param defaultValue the default to be used if the
*    value is not found.
* @return     true if the key's value is false, false if
*    it's true, or the defaultValue if not found.
*/
public boolean isFalseDefault(String key, boolean defaultValue) {
// watch out for this... it is a bit tricky!
    return ! isTrueDefault(key, ! defaultValue);
}
// ----- int methods
/**
* Sets the key value, overwriting any that already exists.
* @param key   the unique key, which may not be null.
* @param value the int value.
*/
public void setInt(String key, int value) {
    setSubclass(key, new Integer(value));
}
/**
* Returns the int value for the key.
* @param key  the key to be used for the lookup.
* @return     the key's value or an exception if not found.
*/
public int getInt(String key) {
    // optimistic implementation :)
    try {
        return ((Integer) getSubclass(key)).intValue();
    } catch (Exception ex) {
        handleTypeException(ex, key, "int");
        return 0;
    }        
}
/**
* Same as getInt(key) except allows a default to be
* returned rather than an exception if not found.
* @param key  the key to be used for the lookup.
* @param defaultValue the default to be used if not found.
* @return the key's value or the defaultValue if not found.
*/
public int getIntDefault(String key, int defaultValue) {
    Object value = getSubclass(key);
    if (value != null && value instanceof Integer) {
        return ((Integer) value).intValue();
    } else {
        return defaultValue;
    }
}
// ----- long methods
/**
* Sets the key value, overwriting any that already exists.
* @param key   the unique key, which may not be null.
* @param value the long value.
*/
public void setLong(String key, long value) {
    setSubclass(key, new Long(value));
}
/**
* Returns the long value for the key.
* @param key  the key to be used for the lookup.
* @return     the key's value or an exception if not found.
*/
public long getLong(String key) {
    // optimistic implementation :)
    try {
        return ((Long) getSubclass(key)).longValue();
    } catch (Exception ex) {
        handleTypeException(ex, key, "long");
        return 0;
    }        
}
/**
* Same as getLong(key) except allows a default to be
* returned rather than an exception if not found.
* @param key  the key to be used for the lookup.
* @param defaultValue the default to be used if not found.
* @return the key's value or the defaultValue if not found.
*/
public long getLongDefault(String key, long defaultValue) {
    Object value = getSubclass(key);
    if (value != null && value instanceof Long) {
        return ((Long) value).longValue();
    } else {
        return defaultValue;
    }
}
// ----- float methods
/**
* Sets the key value, overwriting any that already exists.
* @param key   the unique key, which may not be null.
* @param value the float value.
*/
public void setFloat(String key, float value) {
    setSubclass(key, new Float(value));
}
/**
* Returns the float value for the key.
* @param key  the key to be used for the lookup.
* @return     the key's value or an exception if not found.
*/
public float getFloat(String key) {
    // optimistic implementation :)
    try {
        return ((Float) getSubclass(key)).floatValue();
    } catch (Exception ex) {
        handleTypeException(ex, key, "float");
        return 0;
    }        
}
/**
* Same as getFloat(key) except allows a default to be
* returned rather than an exception if not found.
* @param key  the key to be used for the lookup.
* @param defaultValue the default to be used if not found.
* @return the key's value or the defaultValue if not found.
*/
public float getFloatDefault(String key, float defaultValue) {
    Object value = getSubclass(key);
    if (value != null && value instanceof Float) {
        return ((Float) value).floatValue();
    } else {
        return defaultValue;
    }
}
// ----- double methods
/**
* Sets the key value, overwriting any that already exists.
* @param key   the unique key, which may not be null.
* @param value the double value.
*/
public void setDouble(String key, double value) {
    setSubclass(key, new Double(value));
}
/**
* Returns the double value for the key.
* @param key  the key to be used for the lookup.
* @return     the key's value or an exception if not found.
*/
public double getDouble(String key) {
    // optimistic implementation :)
    try {
        return ((Double)getSubclass(key)).doubleValue();
    } catch (Exception ex) {
        handleTypeException(ex, key, "double");
        return 0;
    }
}
/**
* Same as getDouble(key) except allows a default to be
* returned rather than an exception if not found.
* @param key  the key to be used for the lookup.
* @param defaultValue the default to be used if not found.
* @return the key's value or the defaultValue if not found.
*/
public double getDoubleDefault(String key, double defaultValue) {
    Object value = getSubclass(key);
    if (value != null && value instanceof Double) {
        return ((Double) value).doubleValue();
    } else {
        return defaultValue;
    }
}
//---------- Protected Methods -----------------------------------
/**
* For handline conversion exceptions.
*/
protected void handleTypeException(Exception ex, String key, String type) {
    if (ex instanceof ClassCastException) {
        throw new IllegalStateException("Stored value was not a " + type);
      
    } else if (ex instanceof NullPointerException) {
        if  (containsKeySubclass(key)) {
            throw new IllegalStateException("Stored value was null");
        } else {
            throw new IllegalStateException("Key "+key+" has no associated value");
        }
    } else {
        throw new RuntimeException(ex.getMessage() +
            "\nProblem with key '" + key + "' of type " + type + ".");
    }
}
//---------- Standard --------------------------------------------
private static void print(String text) {
    System.out.println("ConvenientStringMapStd" + text);
}

} // End class