package uhr.core.structure;

import java.util.Enumeration;
import java.util.Hashtable;

/**
* This class represents a single node in a hierarchical
* System Tree. 
* <p>
* <b>The part name "MyCell" is not allowed</b>. This is reserved for use
* in <code>Cell.findPart(String name)</code>.
* <p>
* See <a href="uhr.core.structure.Node.html">full documentation</a>.
*
* @author Jack Harich
*/
public class NodeStd implements Node {

//---------- Protected Fields ------------------------------------
protected String    name;
protected Object    part;
protected Node      parentNode;
protected Hashtable properties = new Hashtable();

//---------- Node Implementation ---------------------------------
//----- name
/**
* Sets the node or part name. This must be unique per container.
* It must be alphanumeric, may contain periods, and must
* not contain blanks or slashes. It cannot be null or a
* zero length String. This is a required property.
* <p>
* Our convention is to capatalize the first letter of each
* word making up the self-documenting part name. Examples
* are "Root", "ControlRegistry" and "UserSecurity".
* <p>
* If part names are used for finding the part in other
* containers, then to avoid name space conflicts the names
* need careful design. This can achieve branch or system
* uniqueness. This will happen if service parts are used
* a lot. To support unique naming we allow periods in the
* name, such as "org.jcon.security.UserAuthorization".
* <p>
* <b>The part name "MyCell" is not allowed</b>. This is reserved for use
* in <code>Cell.findPart(String name)</code>.
*
* @param name  the name of the node.
* @exception IllegalArgumentException  if the name is "MyCell".
*/
public void setName(String name) {
    // TODO - More validation
    if (name.equals("MyCell")) {
        throw new IllegalArgumentException("MyCell is a reserved name.");
    }
    this.name = name;
}
/**
* Returns the part name.
* @return  the name of the part.
*/
public String getName() {
    return name;
}
//----- part
/**
* Sets the node's part, which may be any object whatsoever.
* The object should be unique per container and probably
* unique per System Tree. This is a required property.
* <p>
* If the part implements NodeUser then setNode(Node) will
* be called, using "this" node.
* @param part  the node's part.
*/
public void setPart(Object part) {
    this.part = part;
    
    if (part instanceof NodeUser) {    
        ((NodeUser)part).setNode(this);
    }
}
/**
* Returns the node's part.
* @return  the node's part.
*/
public Object getPart() {
    return part;
}
//----- parentNode 
/**
* Sets the node's parent node. This is required for 
* all nodes in a tree except the root, which is null.
* @param parentNode  the parent of this node or null if
*                    this is the root node.
*/
public void setParentNode(Node parentNode) {
    this.parentNode = parentNode;
}
/**
* Returns the node's parent node or null if this is
* the root.
* @return  the parent of this node or null if this is the root node.
*/
public Node getParentNode() {
    return parentNode;
}
//---------- Properties
/**
* Determines whether the node has the named property. Note that 
* often the property value is not needed, just whether the node
* has it or not.
*
* @param name  the property's name.
* @return  true if the node has the property, false if not.
*/
public boolean hasProperty(String name) {
    return properties.containsKey(name);
}
/**
* Sets the property using the name and value.
*
* @param name  the property name.
* @param value  the property value, which cannot be null.
* @return  the previous value for the name or null if none.
*/
public Object setProperty(String name, Object value) {
    return properties.put(name, value);
}
/**
* Removes the property with the name.
*
* @param name  the property name.
* @return  the value removed or null if none.
*/
public Object removeProperty(String name) {
    return properties.remove(name);
}
/**
* Returns the property with the name or null if not found.
*
* @param name  the property name of interest.
* @return  the property with the name or null if not found. 
*/
public Object getProperty(String name) {
    return properties.get(name);
}
/**
* Returns an Enumeration of all the property names.
*
* @return  the Enumeration of all the property names.
*/
public Enumeration getPropertyNames() {
    return properties.keys();
}
//----- Other
/**
* Returns the root node, which has no parent node.
* @return  the root node.
*/
public Node getRootNode() {
    if (parentNode == null) {
        return this;
    } else {
        return parentNode.getRootNode();
    }
}
/**
* Returns a String representation of the node. The standard
* is the name of the node.
* @return  a String representation of the node.
*/
public String toString() {
    return name;
}

} // End class