package uhr.core.structure;

import java.util.Enumeration;

/**
* This interface represents a single node in a hierarchical
* System Tree. Each node has one part, so this tree defines the 
* plugpoints available for parts to fill.
* <p>
* A node can have a container or leaf part. Containers hold zero 
* or more nodes. The terms part and node are fairly interchangable.
* <p>
* Responsibilites are to maintain the state that applies
* to the plugpoint structure only. This is the node's name, part,
* and parent node. That's all. The design
* philosophy is a Node should remain small and simple, serving
* only a "plugpoint structure role", with nearly no behavior.
* <p>
* But name, part and parent node are not enough.
* The systen or tool designer will eventually need to associate various
* properties with parts. Rather than have the part carry that
* data, or a custom parallel tree structure carry it, a Node 
* has simple named properties. 
* <p>
* Properties are used by core parts or tools, and possible a few
* domain parts. A basic set of "abilities"
* is all the core needs. The keys for these all start with
* "core." Other properties should have a differnt key namespace.
* The basic abilities are described in 
* <a href="reference/NodeBasicAbilities.html">Node Basic Abilities</a>.
* <p>
* We hope we haven't opened up a Pandora's Box with Node properties.
* They are to be used wisely. :-) 
*
* @author Jack Harich
*/
public interface Node {

//---------- Public Methods --------------------------------------

//---------- name 
/**
* Sets the node or part name. This must be unique per container.
* It must be alphanumeric, may contain periods, and must
* not contain blanks or slashes. It cannot be null or a
* zero length String. This is a required property.
* <p>
* The name "MyCell" is not allowed. This is reserved for use
* in Cell.findPart(String name).
* <p>
* Our convention is to capatalize the first letter of each
* word making up the self-documenting part name. Examples
* are "Root", "ControlRegistry" and "UserSecurity".
* <p>
* If part names are used for finding the part in other
* containers, then to avoid name space conflicts the names
* need careful design. This can achieve branch or system
* uniqueness. This will happen if service parts are used
* a lot. To support unique naming we allow periods in the
* name, such as "org.jcon.security.UserAuthorization".
* @param name  the name of the node.
*/
public void setName(String name);

/**
* Returns the part name.
*
* @return  the name of the part.
*/
public String getName();

//---------- part 
/**
* Sets the node's part, which may be any object whatsoever.
* The object should be unique per container and probably
* unique per System Tree. This is a required property.
* <p>
* If the part implements NodeUser then setNode(Node) will
* be called, using "this" node.
* @param part  the node's part.
*/
public void setPart(Object part);

/**
* Returns the node's part.
* @return  the node's part.
*/
public Object getPart();

//---------- paremtNode
/**
* Sets the node's parent node. This is required for 
* all nodes in a tree except the root, which is null.
* @param parentNode  the parent of this node or null if
*                    this is the root node.
*/
public void setParentNode(Node parentNode);

/**
* Returns the node's parent node or null if this is
* the root.
* @return  the parent of this node or null if this is the root node.
*/
public Node getParentNode();

//---------- Properties
/**
* Determines whether the node has the named property. Note that 
* often the property value is not needed, just whether the node
* has it or not.
*
* @param name  the property's name.
* @return  true if the node has the property, false if not.
*/
public boolean hasProperty(String name);

/**
* Sets the property using the name and value.
*
* @param name  the property name.
* @param value  the property value, which cannot be null.
* @return  the previous value for the name or null if none.
*/
public Object setProperty(String name, Object value);

/**
* Removes the property with the name.
*
* @param name  the property name.
* @return  the value removed or null if none.
*/
public Object removeProperty(String name);

/**
* Returns the property with the name or null if not found.
*
* @param name  the property name of interest.
* @return  the property with the name or null if not found. 
*/
public Object getProperty(String name);

/**
* Returns an Enumeration of all the property names.
*
* @return  the Enumeration of all the property names.
*/
public Enumeration getPropertyNames();

//---------- Other
/**
* Returns the root node, which has no parent node.
* @return  the root node.
*/
public Node getRootNode();

/**
* Returns a String representation of the node. The standard
* is the name of the node.
* @return  a String representation of the node.
*/
public String toString();



} // End interface