package uhr.core.role;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

/**
* This class describes the relevant state machine structure.
* This information is used to more effectively use the 
* StateCommandable class. The info may be static or dynamic, but
* we expect most to be static and so have no removals and such.
* For dynamic use create a new StateCommandableInfoStd class having
* the proper data for that point in time.
* <p>
* First add all transitions, then add propagated and default 
* commands. This allows the implementation to check that the
* propagated and default commands exist. 
* <p>
* After making all mutations perform the <code>makeReadOnly()</code>
* method.
* <p>
* This implementation has <b>no validation checks</b> for integrity,
* other than asserting against duplicate adds, and valid propagated
* and default commands. The client has the responsiblity of 
* ensuring the integrity of this class's information, namely that
* the state transition table makes sense.
* 
* @author Jack Harich
*/
public class StateCommandableInfoStd implements StateCommandableInfo {

//---------- Internal Fields -------------------------------------
// Object = String[] 0 to 2, state, command, newState
protected Vector rows = new Vector();

// Object - String command in order added
protected Vector commands = new Vector();

// Object - String state in order added
protected Vector states = new Vector();

// Object - String propagated command in order added
protected Vector propagatedCommands = new Vector();

// Key = String state, Object = String default command
protected Hashtable defaultCommands = new Hashtable();

// Key = String state
// Object = Vector of allowable commands for that state
protected Hashtable allowableCommands = new Hashtable();

protected boolean isDynamic;
protected boolean isReadOnly;

//---------- StateCommandableInfo Implementation -----------------

//----- Mutators
/**
* Adds one row to the state transition table. If in the state, the
* command would cause a transition to the newState. States and 
* commands should
* be well formed names with no slashes, commas or spaces. Periods
* are okay. By convention start them with a capital letter, such
* as "Start" or "Running". Naturally all states and commands are 
* case sensitive. Avoid duplicate adds.
* <p>
* <b>The first transition added must be the start state.</b>
* <p>
* Sample arguments are ("Created", "Init", "Initialized"). None may
* be null.
* <p>
* The commands should be added in a pleasing, informative, 
* consistent order. They will be shown to users in that order.
*
* @param state     the state under consideration.
* @param command   the command received in the state.
* @param newState  the new state the command would cause a
*                  transition to if in the state.
*/
public void addTransition(String state, String command, 
                                                String newState) {
    mutationRequested(); 
                                                                    
    // Add one row
    rows.addElement(new String[] {state, command, newState});
    // Add state if newly encountered
    if (! states.contains(state)) states.addElement(state);
    // Add command if newly encountered
    if (! commands.contains(command)) commands.addElement(command);    
    // Add allowable command if newly encountered
    if (! allowableCommands.containsKey(state)) {
        allowableCommands.put(state, new Vector());
    }
    Vector stateCommands = (Vector)allowableCommands.get(state);
    if (! stateCommands.contains(command)) {
        stateCommands.addElement(command);
    }
}
/**
* Adds a propagated command. This is relevant only for containers.
* Propagated commands are propagated to children who have that
* command, and to their children, etc. For example "Close" is
* a good propagated command. It will allow all parts needing to
* release resources to do so when their Close command is called.
* In this manner large complex systems can more easily behave
* correctly.
* <p>
* We assume this is unrelated to state. If this turns out to be
* false, we will add <code>addPropagatedCommand(state, command)</code>.
* 
* @param command  the propagated command.
*/
public void addPropagatedCommand(String command) {
    mutationRequested();    
    
    if (! commands.contains(command)) {
        throw new IllegalArgumentException("The command '" +
        command + "' is not in the state transition table.");
    }
    if (propagatedCommands.contains(command)) {
        throw new IllegalStateException("The propagated command '"
        + command + " has already been added.");
    } else {
        propagatedCommands.addElement(command);
    }
}
/**
* Adds the default command for a state. This the most sensible
* command to do when in that state. It's designed to allow a
* StateCommandable to be "cycled through" from beginning to final
* state, or at least back to whare it stated.
*
* @param state    the state in question.
* @param command  the default command for the state.
*/
public void addDefaultCommand(String state, String command) {
    mutationRequested();
    
    if (! states.contains(state)) {
        throw new IllegalArgumentException("The state '" +
        state + "' is not in the state transition table.");
    }    
    if (! commands.contains(command)) {
        throw new IllegalArgumentException("The command '" +
        command + "' is not in the state transition table.");
    }
    if (defaultCommands.containsKey(state)) {
        throw new IllegalStateException("The state '"
        + state + " has already been added to default commands.");
    } else {
        defaultCommands.put(state, command);
    }
}
/**
* Sets whether this information is static or dynamic.
* The default if false, which is most cases.
* @param isDynamic  true if dynamic, false if not.
*/
public void setDynamic(boolean isDynamic) {
    mutationRequested();
    this.isDynamic = isDynamic;
}
/**
* Makes the information read only. This allows an instance to be
* safely reused and passed to whomever needs to read it. This should
* be called after all mutations are done, after which no more
* mutations are allowed.
* <p>
* This method itself is a mutation, so it may only be called once.
*/
public void makeReadOnly() {
    mutationRequested();
    isReadOnly = true;
}
//----- Readers
/**
* Determines whether this information is dynamic or not.
* @return  true if dynamic, false if not.
*/
public boolean isDynamic() {
    return isDynamic;
}
/**
* Determines whether this information is read only.
* @return  true if read only, false if mutations are still allowed.
*/
public boolean isReadOnly() {
    return isReadOnly;
}
/**
* Returns all the commands in the order added.
* @return  an enumeration of all commands.
*/
public Enumeration getAllCommands() {
    return commands.elements();
}
/**
* Returns the allowable commands for the state. All commands not
* included are not allowed.
* @param state  the state in question.
* @return  an enumeration of all allowable commands for the state.
*/
public Enumeration getAllowableCommands(String state) {
    Vector stateCommands = (Vector)allowableCommands.get(state);
    return stateCommands.elements();
}
/**
* Returns true if the command is propagated, false if not or if
* not a valid command.
* @param command  the command in question.
* @return  true if propagated to children, false if not.
*/
public boolean isCommandPropagated(String command) {
    return propagatedCommands.contains(command);
}
/**
* Determines if the command exists, false if not. 
* @param command  the command in question.
* @return  true if it exists, false if not.
*/
public boolean hasCommand(String command) {
    return commands.contains(command);
}
/**
* Returns all posible states.
* @return  an enumeration of all possible states.
*/
public Enumeration getAllStates() {
    return states.elements();    
}
/**
* Returns the default command for the state or null if none.
* @param state  the state in question.
* @return  the default command for the state or null if none.
*/
public String getDefaultCommand(String state) {
    return (String)defaultCommands.get(state);    
}
//----- Reading the table
/**
* Returns the number of rows in the state transition table. 
* @return  the number of rows in the state transition table.
*/
public int getTableRowCount() {
    return rows.size();
}
/**
* Returns one row in the state transition table.
* @param rowNumber  the row number to return. Zero based is used.
* @return  a String array holding one table row. Array indexes
* 0, 1, 2 refer to table columns "state", "command", "newState".
* @exception IllegalArgumentException if the rowNumber is invalid.
*/
public String[] getTableRow(int rowNumber) {
    if (rowNumber < 0 || rowNumber >= rows.size()) {
        throw new IllegalArgumentException("Invlalid rowNumber. " +
        "The state transition table contains " + rows.size() + " rows.");
    } else {
        return (String[])rows.elementAt(rowNumber);
    }
}
/**
* Returns a list of all rows in the table in the order added.
* @return  a new line delimited list of all rows in the table
*          or null if the table is empty.
*/
public String listStateTransitionTable() {
    if (rows.isEmpty()) return null;
    
    String text = "";
    for (int i = 0; i < rows.size(); i++) {
        String[] row = (String[])rows.elementAt(i);
        text += row[0] + ", " + row[1] + ", " + row[2];
        if (i < rows.size()) text += "\n";
    }
    return text;
}
//---------- Protected Methods -----------------------------------
protected void mutationRequested() {
    if (isReadOnly) throw new IllegalStateException(
    "Cannot make any more mutations. This information has been " +
    "set to read only.");
}
//---------- Standard --------------------------------------------
private static void print(String text) {
    System.out.println("StateCommandableInfoStd" + text);
}

} // End class