package uhr.core.role;

import uhr.core.tron.Infotron;

/**
* This interface allows the state of an class to be changed by
* commands. Means are avaiable for publishing information
* about the state machine, allowing the designer, user and other
* classes to better understand things.
* <p>
* For example one may wish to control the lifecycle of parts,
* relieving them of that confusing burden and allowing a part 
* framework to achieve higher reuse.
* <p>
* One planned use for this is to allow users to right click on a
* System Tree part node. The context menu is dynamically built
* using the current state and StateCommandableInfo. The current
* state is mentioned and all commands are listed. Those commands
* not allowable at this time are disabled. The part designer can
* add unit tests, what ifs, etc for development use, and remove
* them later.
* <p>
* Special thanks to <A HREF="mailto:WayneA@PSSG.com">Wayne Allen</A>
* for suggesting the default command trick, which he has used 
* in his system designs. He has used this to cycle from an object's
* initial to terminal state and other things. We could use it to be
* the default command for a part, such as what happens when a part
* is double clicked.
*
* @author Jack Harich
*/
public interface StateCommandable {

//---------- Public Methods --------------------------------------

/**
* The class should perform the command if its present state
* allows it.
*
* @param command  the command to perform. It must be in the
*                 published list of commands available.
*
* @exception  IllegalArgumentException if unknown command. 
* @exception  IllegalStateException if not in proper state to
*             perform the command. Gee, this is precise. :-)
* @exception  RuntimeException or such if the command fails.
*/
public void doStateCommand(String command);

/**
* Returns the current state, which is never null.
*/
public String getCurrentState();

/**
* Returns information about the StateCommandable's business rules
* related to its state transition table.
* <p>
* Since this is used for read only purposes, the info objects
* <b>may be shared</b> between StateCommandables. This is the 
* Flyweight Pattern, and can modestly reduce resource requirements
* and increase startup speed.
* <p>
* Most StateCommandables will have static StateCommandableInfo.
* They are assumeed to return the same info instance each time
* with the identical data. This allows optimizing assumptions.
* If the info is dynamic then a new StateCommandableInfo should be
* returned each time.
*
* @return  the information about the StateCommandable.
*/
public StateCommandableInfo getStateCommandableInfo();

} // End interface


//================================================================
// NOT USED, DELETE LATER - JH
/**
* Returns structured information describing the state transition
* table used. Note this can be created from DK. Rather than 
* tediously define the format we give an example: 
* <pre>
* // These should be in a pleasing, informative, consistent order.
* AllCommands hasLines:
*     Init
*     Start
*     Stop
*     Close
*     End: AllCommands
* //
* // These are for containers only. This is an optional property.
* PropogatedCommands hasLines:
*     Start
*     End: PropogatedCommands
* //
* // This should allow going from initial to final state, or
* // back to the initial state. They should be the command a user
* // would want to perform on the object if it was in that state
* // and they wanted to see it do its normal thing.
* // Final states should have no comma and no command.
* AllStatesAndDefaultCommands hasLines:
*     Created,     Init
*     Initialized, Start
*     Running,     Close
*     Stopped,     Start
*     Closed
*     End: AllStatesAndDefaultCommands
* //
* // Table has "State, Command, NewState" per line.
* StateTransitionTable hasLines:
*     Created,     Init,  Initialized
*     Initialized, Start, Running
*     Running,     Stop,  Stopped
*     Running,     Close, Closed
*     Stopped,     Start, Running
*     End: StateTransitionTable
* </pre>
*
* @return  the information describing the state machine used.
*/

