package uhr.core.role;

import uhr.core.tron.Infotron;

/**
* This allows a part to be replaced with a new part at runtime. 
* This ability is crucial if we are to achieve system uptime of 24 
* hours a day, 365 days a year. The ability to trivially replace 
* parts while a system is running will open up a new breed of 
* systems - living systems, who stay alive indefinitely, growing, 
* mutating, merging, spawning offspring, being repaired, learning, 
* moving from old to new hardware, etc.
* <p>
* Parts implementing this interface should be designed to easily
* replace themselves using these methods. Note how we encourage
* indefinite replacement by defining how replaceable state is
* smoothly passed from each generation to the next.
* <p>
* In conjunction with this interface some parts may need to
* implement <code>StateCommandable</code> so that they can pause
* and resume while being replaced. Replacement is a normal 
* lifecycle step.
* <p>
* This interface is used by tools, not this layer. The interface is
* in this layer merely so all parts, even core parts, can 
* implement it. A tool part such as PartReplacer would manipulate
* core parts to do the replacement algorithm steps. A sophisticated
* PartReplacer would include features such as rollback, replacing 
* client Direct Links gained via Part User, replacing entire 
* branches, replacing all instances of a part, and DK updates. 
* These are not shown or designed yet, but it all hinges on this
* interface.
* <p>
* There are some related modifications needed to MessageRouter, 
* Cell and such to support this algorithm, which was added recently.
* For example Cell may need 
* <code>replacePart(String name, Object newPart)</code>, which 
* would combine the Remove and Add algorithm steps. We may add the
* same method to MessageRouter, plus 
* <code>pausePartMessages(String partName)</code> and 
* <code>resumePartMessages(String partName)</code>. Or to keep it
* simple we may just have the router do pause and resume.
* <p>
* The algorithm for part replacement is quite interesting, and 
* takes full advantage of Messages and other features. Imagine 
* trying to do this generically in a system without UHR. :-)
* <p>
<BLOCKQUOTE>
    <P><IMG SRC="images/blue-ball.gif" WIDTH="12" HEIGHT="12" ALIGN="BOTTOM" BORDER="0"> 
    Pause Old Part if supported</P>
    <BLOCKQUOTE>
        <P><IMG SRC="images/blue-ball.gif" WIDTH="12" HEIGHT="12" ALIGN="BOTTOM" BORDER="0"> 
        Disconnect Messages on Old Part</P>
        <BLOCKQUOTE>
            <P><IMG SRC="images/blue-ball.gif" WIDTH="12" HEIGHT="12" ALIGN="BOTTOM" BORDER="0"> 
            Remove Old Part from Cell</P>
            <BLOCKQUOTE>
                <P><IMG SRC="images/blue-ball.gif" WIDTH="12" HEIGHT="12" ALIGN="BOTTOM" BORDER="0"> <FONT SIZE="4"
                COLOR="#00CC00">Transfer state from Old to New Part</FONT></P>
            </BLOCKQUOTE>
            <P><IMG SRC="images/blue-ball.gif" WIDTH="12" HEIGHT="12" ALIGN="BOTTOM" BORDER="0"> 
            Add New Part to Cell</P>
        </BLOCKQUOTE>
        <P><IMG SRC="images/blue-ball.gif" WIDTH="12" HEIGHT="12" ALIGN="BOTTOM" BORDER="0"> 
        Connect Message on New Part</P>
    </BLOCKQUOTE>
    <P><IMG SRC="images/blue-ball.gif" WIDTH="12" HEIGHT="12" ALIGN="BOTTOM" BORDER="0"> 
    Resume New Part if supported</P>
</BLOCKQUOTE>
* Transfer state is done without either part knowing about the 
* other, a beautiful trick. Both parts must implement Replaceable.
* A mediator part (such as PartReplacer) calls 
* <code>oldPart.getReplaceableState()</code> and then 
* <code>newPart.setReplaceableState(infotron)</code>.  Infotron is
* a type of Datatron. This is a case of Datatrons at their finest, 
* flowing data from one part to another without the parts ever 
* knowing what kind of sneaky game is up! :-)
*
* @author Jack Harich
*/
public interface Replaceable {

//---------- Public Methods --------------------------------------

/**
* This is called on the old part. It should provide the state
* that is relevant to being replaced, and release all references
* in preparation for garbage collection.
* <p>
* This method is called first, before <code>setReplaceableState(Infotron)</code>.
*
* @return  the replaceable state.
*/
public Infotron getReplaceableState();


/**
* This is called on the new part. It provides the state it needs
* to initialize itself to start running as a replacement.
* <p>
* In case the new part objects to being replaced, we can call this
* method on the old part. It should restore itself to normal
* state. If it released references it can get them back from the
* newState, which in this case is the old state. A part doesn't
* know if it's an old or new part, so this works exactly the same
* for both in a transparent manner. Beautiful....
*
* @param newState  the new replaceable state the part should use
*                   to initialize itself.
*
* @return  null if okay or an explaination of why the replacement
*          cannot be made at this time.
*/
public String setReplaceableState(Infotron newState);






} // End interfacext.