package uhr.core.role;

import java.util.Enumeration;

/**
* This interface allows a class to have delegates. A delegate
* performs part of a class's responsibilities. <b>The strategic goal
* is to allow more responsibility in a single class without causing
* more complexity</b>. This is accomplished by the delegate being
* separately encapsulated. Furthermore, the class may provide a
* delegate or it may be provided for it.
* <p>
* This provides the Exposed Delegator Pattern for parts. This 
* pattern allows clients to get and use a part's Delegate for 
* certain responsiblities. This greately reduces the size and 
* complexitiy of some parts, allows loose coupling between the 
* Delegate and it's host, etc. Note how the host can provide its
* own Delegate, or it can be set by a Mediator. This pattern allows
* a slew of radical framework improvements.
* <p>
* One example is a Cell has a "StateCommandable" Delegate which is
* the CellLifecycle. The Cell never even knows it has this Delegate.
* <p>
* A great advantage is when looping through parts in a collection 
* to check for a certain interface, say InterfaceA. As Steve 
* pointed out, by doing the Delegate check first we allow the 
* Delegate to shadow the part's InterfaceA. The algorithm is:
* <BLOCKQUOTE>
* 1. If the part implements Delegator and has the InterfaceA 
* Delegate, use it.
* <p>
* 2. If the part implements the InterfaceA, use that.
* <p>
* 3. Otherwise assume the part doesn't support InterfaceA.
* </BLOCKQUOTE>
* <p>
* The greatest good is achieved when delegates are provided for a
* class and the class never even knows they're there. These
* "hidden" delegates usually have a reference to the delegator. The
* delegator should not use them for anything, and pretend they're 
* not even there.
* <p>
* Hidden delegates should be kept in something like a Hashtable.
* Since we are planning for mostly this, we return an <code> 
* Enumeration</code> instead of a <code>String[]</code> for the
* delegate names. This makes the implementation easier.
*
* @author Jack Harich
*/
public interface Delegator {

//---------- Public Methods --------------------------------------

/**
* Returns the named delegate or null if none. This may safely be
* used instead of a method like: <pre>
*     public boolean hasDelegate(); </pre>
* which would be redundant and a perfromance hit since if true you
* must get the delegate anyhow.
*
* @param delegateName  the name of the desired delegate.
* @return              the named delegate or null if none.
*/
public Object getDelegate(String delegateName);


/**
* Sets the named delegate.
*
* @param delegateName  the name of the delegate, which must be
*                      unique within the Delegator.
* @param delegate      the delegate instance.
*/
public void setDelegate(String delegateName, Object delegate);


/**
* Returns an enumeration of all the delegate names in no
* particular order.
* 
* @return  an enumeration of the delegate names.
*/
public Enumeration getDelegateNames();


} // End interface