package uhr.core.msg;

import uhr.core.role.Replicatable;
import uhr.core.role.StateCommandable;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

/**
* This routes Messages from senders to receivers, handles 
* connections and provides related services such as for tools 
* needing to know things about circuits. Messages are multicast
* events used for part Anonymous Collaboration.
* <p>
* See <a href="uhr.core.msg.MessageRouter.html">full documentation</a>.
* This is NOT a thread safe implementation.
* <p>
* IMPLEMENTATION STATUS - Monitor not yet implemented. Service
* message signals and interests not published. Tool methods are 
* undesigned and unimplemented. 
*
* @author Jack Harich
*/
public class MessageRouterStd implements MessageRouter {

//---------- Internal Fields -------------------------------------
//--- Sender and receiver connections 
// Key = messageName + "/" + senderName, Object = Connection
protected Hashtable signalOutConnections   = new Hashtable();
protected Hashtable signalInConnections    = new Hashtable();
protected Hashtable serviceNeedConnections = new Hashtable();
// Key = messageName + "/" + receiverName, Object = Connection
protected Hashtable interestOutConnections     = new Hashtable();
protected Hashtable interestInConnections      = new Hashtable();
protected Hashtable serviceSupplierConnections = new Hashtable();

// Receivers
// Key = messageName, Object = Vector of MessageReceivers
protected Hashtable interestOutReceivers = new Hashtable();
protected Hashtable interestInReceivers = new Hashtable();
// Key = messageName, Object = MessageReceiver, one max allowed
protected Hashtable serviceSupplierReceivers = new Hashtable();

//--- Direct connections
// Key = messageName + "/" + senderName + "/" + receiverName,
//   Object = Connection
protected Hashtable directConnections = new Hashtable();

// Key = messageName + "/" + senderName, 
//   Object = Vector of MessageReceivers
protected Hashtable directReceivers = new Hashtable();

//----- Other
protected MessageReceiver myReceiver; // null if root
protected boolean         isMonitoring;
protected MessageRouter   parentRouter; // null if we are the root
protected Vector          emptyList = new Vector(); // Always empty

//---------- MessageRouter Implementation ------------------------
//----- Connections
/**
* This creates a connection. 
*
* @param connection  the connection to connect.
*/
public void connect(Connection connection) {
    String messageName = connection.getMessageName();
    //***String key = messageName + "/" + connection.getPartName();
    String key = createConnectionsKey(connection);
    Hashtable connections = getConnectionsFor(connection);
    
    // Assert connection has not already been made
    if (connections.containsKey(key)) {
        throw new IllegalStateException("The connection has already " +
        "been made for messageName = '" + messageName + "', " +
        "senderName = '" + connection.getSenderName() + "', " +
        "receiverName = '" + connection.getReceiverName() + "', " +
        "connectionType = '" + connection.getConnectionTypeString() + "'.");
    }
    // Save connection for a rainy day :-)
    connections.put(key, connections);
    
    // Special cases. No need to check for dupes due to above assertion.
    if (connection.getConnectionType() == Connection.INTEREST_OUT) {
        connectReceiver(connection, interestOutReceivers);
    }
    if (connection.getConnectionType() == Connection.INTEREST_IN) {
        connectReceiver(connection, interestInReceivers);
    }
    if (connection.getConnectionType() == Connection.DIRECT) {
        connectReceiver(connection, directReceivers);
    }    
    if (connection.getConnectionType() == Connection.SERVICE_SUPPLIER) {
        // Assert max of one supplier per message name
        if (serviceSupplierReceivers.containsKey(messageName)) {
            throw new IllegalStateException("A message service has " +
            "already been connected for messageName = '" + 
            messageName + "'."); // Need more info - TODO
        }
        serviceSupplierReceivers.put(messageName, connection.getMessageReceiver());
    }
}
/**
* This disconnects a connection previously made with <code>
* connect(Connection)</code>.
*
* @param connection  the connection to disconnect.
* @return  the connection removed or null if none.
*/
public Connection disconnect(Connection connection) {
    String messageName = connection.getMessageName();
    // **String key = messageName + "/" + connection.getPartName();
    String key = createConnectionsKey(connection);
    Hashtable connections = getConnectionsFor(connection);
    
    // Assert connection exists
    // TODO - We should also have a Connection.equals() check
    Connection oldConnection = (Connection)connections.get(key);
    if (oldConnection == null) return null;
    
    // Remove connection
    connections.remove(key);
    // Remove special cases
    if (connection.getConnectionType() == Connection.INTEREST_OUT) {
        disconnectReceiver(connection, interestOutReceivers);
    }
    if (connection.getConnectionType() == Connection.INTEREST_IN) {
        disconnectReceiver(connection, interestInReceivers);
    }
    if (connection.getConnectionType() == Connection.DIRECT) {
        disconnectReceiver(connection, directReceivers);
    }    
    if (connection.getConnectionType() == Connection.SERVICE_SUPPLIER) {
        serviceSupplierReceivers.remove(messageName);
    }
    return oldConnection; 
}
protected String createConnectionsKey(Connection connection) {
    // The order of these could be optimized
    
    if (connection.isSenderConnection()) {
        return connection.getMessageName() + "/" +
               connection.getSenderName();
               
    } else if (connection.isReceiverConnection()) {
        return connection.getMessageName() + "/" +
               connection.getReceiverName();
    } else {
        return connection.getMessageName() + "/" +
               connection.getSenderName()  + "/" +
               connection.getReceiverName(); 
    }
}
protected void connectReceiver(Connection connection, 
                               Hashtable hashReceivers) {
    
    String key = createReceiversKey(connection);
    Vector receivers = (Vector)hashReceivers.get(key);
    if (receivers == null) {
        receivers = new Vector();
        hashReceivers.put(key, receivers);
    }
    receivers.addElement(connection.getMessageReceiver());
}
protected void disconnectReceiver(Connection connection, 
                                  Hashtable hashReceivers) {
    
    String key = createReceiversKey(connection);                                
    Vector receivers = (Vector)hashReceivers.get(key);
    if (receivers != null) {
        receivers.removeElement(connection.getMessageReceiver());
        if (receivers.isEmpty()) hashReceivers.remove(key);
    }
}
protected String createReceiversKey(Connection connection) {
    if (connection.isDirectConnection()) {
        return connection.getMessageName() + "/" +
               connection.getSenderName();
    } else {                                      
        return connection.getMessageName();                         
    }
}
protected Hashtable getConnectionsFor(Connection connection) {
    switch (connection.getConnectionType()) {
        case Connection.SIGNAL_OUT:
            return signalOutConnections;
        case Connection.SIGNAL_IN:
            return signalInConnections;
        case Connection.SERVICE_NEED:
            return serviceNeedConnections;                      
        case Connection.INTEREST_OUT:
            return interestOutConnections;
        case Connection.INTEREST_IN:
            return interestInConnections;
        case Connection.SERVICE_SUPPLIER:
            return serviceSupplierConnections;
        case Connection.DIRECT:
            return directConnections;
        default:
            throw new IllegalArgumentException(
            "Unknown connection type '" + connection.getConnectionType() + "'.");
    }    
}
//----- Send
/**
* Sends the message from the sender on to its destinations.
* See <a href="uhr.core.msg.MessageRouter.html#sendMessage(uhr.core.msg.Message, uhr.core.msg.MessageSender)">method documentation</a>.
*
* @param message     the message to send.
* @param sender      the original sender of the message.
* @param senderName  the name of the sender part.
*/
// ****** mod others to senderName
public void sendMessage(Message message, MessageSender sender,
                                         String senderName) {

    // TODO - Monitoring not implemented.
    String messageName = message.getName();
    String key = messageName = "/" + senderName;
    
    // 1. Internally determine the connections existing for the message
    // name and sender. Put the results in isSignalOut, isSignalIn and
    // isServiceNeed.
    boolean isSignalOut   = signalOutConnections.containsKey(key);
    boolean isSignalIn    = signalInConnections.containsKey(key);
    boolean isServiceNeed = serviceNeedConnections.containsKey(key);
    boolean isDirect      = directReceivers.containsKey(key);
    
    // 2. If isSignalOut is true, call my receiver, as set in my
    // <code>setMessageReiver(MessageReceiver)</code> method.
    if (isSignalOut) myReceiver.receive(message);
    
    // 3. If isSignalIn is true, call all receivers connected with
    // INTEREST_IN for the message name.
    if (isSignalIn) { 
        Vector receivers = (Vector)interestInReceivers.get(messageName);
        if (receivers != null) sendToReceivers(message, receivers);
    }
    // 4. If isDirect is true, call the receivers connected with
    // DIRECT for the message name <b>from that sender</b>. 
    if (isDirect) { 
        Vector receivers = (Vector)directReceivers.get(key);
        if (receivers != null) sendToReceivers(message, receivers);
    }        
    // 5. If isServiceNeed is true:
    if (isServiceNeed) serviceMessage(message, sender);
} 

/**
* Allows a router to handle a "service message". See the algorithm
* in <code>sendMessage(...)</code>
*
* @param message  the message to send.
* @param sender   the original sender of the message.
*/
public void serviceMessage(Message message, MessageSender sender) {
    // (continuation of "5. If isServiceNeed is true:")
    // - If this router has a SERVICE_SUPPLIER for the message 
    // name, call that receiver.
    if (handleServiceNeedHere(message)) {
        // Do nothing. We have handled isServiceNeed.
    
    // - Otherwise if the parent router is not null, call 
    //      parentRouter.serviceMessage(message, sender);
    } else if (parentRouter != null) {
        parentRouter.serviceMessage(message, sender);
    
    // - Otherwise there is no one to handle the service needed, so
    //     a RuntimeException is thrown since this is
    //     a probable configuration error.
    } else {
        throw new RuntimeException("no message service for the " +
        "message name '" + message.getName() + "' is connected.");
    }
}
// Returns true if handled, false if not
protected boolean handleServiceNeedHere(Message message) {
    MessageReceiver receiver = (MessageReceiver)
        serviceSupplierReceivers.get(message.getName());
    if (receiver != null) {
        receiver.receive(message);
        return true;
    } else {
        // No service supplier in this cell, so 
        return false;
    }
}                       
//----- Miscellaneous properties
/**
* Sets the MessageRouter in the parent cell of this router.
* @param parentRouter  the parent router.
*/
public void setParentRouter(MessageRouter parentRouter) {
    this.parentRouter = parentRouter;
}
/**
* Returns the parent router or null if this is the root router.
* @return the parent router.
*/
public MessageRouter getParentRouter() {
    return parentRouter;
}
//----- For tools
// ***** TODO *****
// Perhaps getSignalsOut(), getSignalsIn(), getInterestOut(), getInterestsIn()
/**
* Returns all the message names currently connected.
* @return  an enumeration message names.
*/
public Enumeration getMessageNames() {
    return null; // **************
}
/**
* Returns all receivers for the messageName.
* @return  an enumeration of MessageReceivers.
*/
public Enumeration getMessageReceivers(String messageName) {
    return null; // ****************
}
/**
* Returns all message names the receiver is connected to.
* @return  an enumeration of message names.
*/
public Enumeration getMessageNamesWithReceiver(
                                     MessageReceiver receiver) {
    return null; // *************                                     
}
//---------- MessageReceiver 
/**
* The router receives the message and routes it to its connected
* receivers (<b>InterestOut</b>), or in some cases handles the 
* message itself. The Messages handled by the router itself are: 
* <pre>
*   Message Name   - TurnRouterMonitoringOn
*   Message Name   - TurnRouterMonitoringOff
*   Message Name   - GetRouterMonitoringState
*                        State = true or false
* </pre>
* The initial monitoring state is false, aka turned off. Now, a
* few words about the monitoring feature:
* <p>
* We offer support for animation of live systems and other things 
* via message monitoring. Turning monitoring on will cause the 
* router to begin being a MessageSender for Messages sent when 
* various things happen in the router. Currently these are <b>before</b>
* and <b>after</b> a message is sent to a receiver, excluding the 
* monitor messages. We call the receiver of monitoring messages a 
* "monitor".
* <p>
* For example, to visually show the travel of messages in a System
* Tree, the monitor could highlight the sender node when before
* notification is received, sleep, then paint a line to the 
* receiving node when the after notification is received. 
* <p>
* Note that monitoring will slow message transmission down, so 
* usually it's turned off.
*
* @param message  the message to route.
*/
public void receive(Message message) {
    String messageName = message.getName();
    
    // The router handles these interests itself:
    if (messageName == "TurnRouterMonitoringOn") {
        isMonitoring = true;
    } else if (messageName == "TurnRouterMonitoringOff") {
        isMonitoring = false;
    } else if (messageName == "GetRouterMonitoringState") {
        message.setBoolean("State", isMonitoring);
    }
    // Then it gives everyone else a chance
    // Forward to interestOutConnections 
    Vector receivers = (Vector)interestOutReceivers.get(messageName);
    if (receivers != null) sendToReceivers(message, receivers);
}
//---------- MessageSender
/**
* Sets the receiver to use when sending messages, which for routers
* is the parent router. This allows a router to behave like any
* other part. The Messages sent by the router itself are: 
* <p>
* If monitoring is turned on these messages are sent: <pre>
*   Message Name      - "MonitorBeforeReceive"
*   "Message"         - The Message transmitted.
*   "MessageSender"   - The source of the message.
*   "MessageReceiver" - The destination of the message.
*   "ConnectionType"  - "Out" or "In".
*   //
*   Message Name - "MonitorAfterReceive"
*   (same properties)
* </pre>
* We expect the monitor to be a message service, allowing a single
* part to monitor an entire system. To start monitoring, the
* monitor (or another part, or even a possessor of the root node)
* would walk the System Tree, get each MessageRouter, and send it
* the "TurnRouterMonitoringOn" message.
* 
* @param receiver  the MessageReceiver to use when sending Messages.
*/
public void setMessageReceiver(MessageReceiver receiver) {
    myReceiver = receiver;
}
//---------- Replicatable
/**
* Returns a copy of itself without mission state.
* @return  the replication.
*/
public Object replicate() {
    // No DK, so this is easy
    return new MessageRouterStd();
}
//---------- Protected Methods -----------------------------------
protected void sendToReceivers(Message message, Vector receivers) {
    // Switch to using Vector indexes, faster.
    // Notice how we don't clone the receivers per Bean event spec
    Enumeration enum = receivers.elements();
    while (enum.hasMoreElements()) {
        MessageReceiver receiver = (MessageReceiver)enum.nextElement();
        receiver.receive(message);
    }    
} 
//---------- Standard --------------------------------------------
private static void print(String text) {
    System.out.println("MessageRouterStd" + text);
}

} // End class