package uhr.core.msg;

/**
* This class describes a single connection between parts in a
* cell and the cell's MessageRouter. It is used to connect and
* disconnect connections, as well as describe existing connections.
* It is immutable.
*
* @author Jack Harich
*/
public class ConnectionStd implements Connection {

//---------- Internal Fields -------------------------------------
protected String messageName;
protected int    connectionType;
protected String senderName;
protected String receiverName;

protected MessageSender   sender;
protected MessageReceiver receiver;

//---------- Initialization --------------------------------------
/**
* A no-arg constructor for internal use and future serialization.
*/
public ConnectionStd() {
    // Do nothing
}
//---------- Connection Implementation ---------------------------
//----- Creators
/**
* Creates a sender connection. Senders send message signals.
*
* @param messageName     the name of the signal to connect.
* @param connectionType  the type of connection, which must be 
*                          SIGNAL_OUT, SIGNAL_IN or SERVICE_NEED.
* @param senderName      the name of the sender part.
* @param sender          the sender in the connection.
*/
public Connection createSenderConnection(
                     String        messageName,
                     int           connectionType,
                     String        senderName,
                     MessageSender sender) {
                     
    if (connectionType != Connection.SIGNAL_OUT &&
        connectionType != Connection.SIGNAL_IN  &&                     
        connectionType != Connection.SERVICE_NEED) {
     
        throw new IllegalArgumentException("The connectionType " +
        "must be SIGNAL_OUT, SIGNAL_IN or SERVICE_NEED for this constructor.");    
    }  
    ConnectionStd conn = new ConnectionStd();
    conn.messageName    = messageName;
    conn.connectionType = connectionType;
    conn.senderName     = senderName;
    conn.sender         = sender;
    
    return conn;
}                     
/**
* Creates a receiver connection. Receivers have interests
* in particular messages.
*
* @param messageName     the name of the interest to connect.
* @param connectionType  the type of connection, which must be 
*                          INTEREST_OUT, INTEREST_IN or SERVICE_SUPPLIER.
* @param receiverName    the name of the receiver part.
* @param receiver        the receiver in the connection.
*/
public Connection createReceiverConnection(
                     String          messageName,
                     int             connectionType,
                     String          receiverName,
                     MessageReceiver receiver) {
                     
    if (connectionType != Connection.INTEREST_OUT &&
        connectionType != Connection.INTEREST_IN  &&                     
        connectionType != Connection.SERVICE_SUPPLIER) {
     
        throw new IllegalArgumentException("The connectionType " +
        "must be INTEREST_OUT, INTEREST_IN or SERVICE_SUPPLIER for this constructor.");    
    }  
    ConnectionStd conn = new ConnectionStd();
    conn.messageName    = messageName;
    conn.connectionType = connectionType;
    conn.receiverName   = receiverName;
    conn.receiver       = receiver;
    
    return conn;
} 
/**
* Creates a direct connection. Messages with this 
* messageName will go directly as possible from the sender to
* receiver.
*
* @param messageName     the name of the interest to connect.
* @param senderName      the name of the sender part.
* @param receiverName    the name of the receiver part.
* @param sender          the sender in the connection.
* @param receiver        the receiver in the connection.
*/
public Connection createDirectConnection(
                     String          messageName,
                     String          senderName,
                     String          receiverName,
                     MessageSender   sender,
                     MessageReceiver receiver) {
    
    ConnectionStd conn = new ConnectionStd();                 
    conn.messageName    = messageName;
    conn.connectionType = Connection.DIRECT;
    conn.senderName     = senderName;
    conn.receiverName   = receiverName;
    conn.sender         = sender;
    conn.receiver       = receiver;
    
    return conn;
} 
//----- Readers
/**
* Determines whether this is a sender connection.
* @return  true is this is a sender connection, false if not.
*/
public boolean isSenderConnection() {
    switch (connectionType) {
        case Connection.SIGNAL_OUT:
            return true;
        case Connection.SIGNAL_IN:
            return true;
        case Connection.SERVICE_NEED:
            return true;                      
        default:
            return false;
    }        
}
/**
* Determines whether this is a receiver connection.
* @return  true is this is a receiver connection, false if not.
*/
public boolean isReceiverConnection() {
    switch (connectionType) {                   
        case Connection.INTEREST_OUT:
            return true;
        case Connection.INTEREST_IN:
            return true;
        case Connection.SERVICE_SUPPLIER:
            return true;
        default:
            return false;
    }        
}
/**
* Determines whether this is a direct connection.
* @return  true is this is a direct connection, false if not.
*/
public boolean isDirectConnection() {
    return (connectionType == Connection.DIRECT ? true : false);
}
/**
* Returns the message name.
* @return  the the message name.
*/
public String getMessageName() {
    return messageName;
}
/**
* Returns the connection type.
* @return  the connection type.
*/
public int getConnectionType() {
    return connectionType;
}
/**
* Returns the String for the connection type constant. For example
* if the type is Connection.DIRECT then "DIRECT" is returned.
* @return  the String for the connection type constant.
*/
public String getConnectionTypeString() {

    switch (connectionType) {
        case Connection.SIGNAL_OUT:
            return "SIGNAL_OUT";
        case Connection.SIGNAL_IN:
            return "SIGNAL_IN";
        case Connection.SERVICE_NEED:
            return "SERVICE_NEED";                      
        case Connection.INTEREST_OUT:
            return "INTEREST_OUT";
        case Connection.INTEREST_IN:
            return "INTEWREST_IN";
        case Connection.SERVICE_SUPPLIER:
            return "SERVICE_SUPPLIER";
        case Connection.DIRECT:
            return "DIRECT";            
        default:
            throw new IllegalArgumentException(
            "Unknown connection type '" + connectionType + "'.");
    }    
}
/**
* Returns the connection constant given its String representation.
* For example if given "DIRECT" then Connection.DIRECT is returned.
* @return  the connection constant for the String.
*/
public int convertTypeToConstant(String connectionType) {

    if (connectionType.equals("SIGNAL_OUT")) {
        return Connection.SIGNAL_OUT;
    } else if (connectionType.equals("SIGNAL_IN")) {
        return Connection.SIGNAL_IN;
    } else if (connectionType.equals("SERVICE_NEED")) {
        return Connection.SERVICE_NEED;
    } else if (connectionType.equals("INTEREST_OUT")) {
        return Connection.INTEREST_OUT;
    } else if (connectionType.equals("INTEREST_IN")) {
        return Connection.INTEREST_IN;
    } else if (connectionType.equals("SERVICE_SUPPLIER")) {
        return Connection.SERVICE_SUPPLIER;
    } else if (connectionType.equals("DIRECT")) {
        return Connection.DIRECT;
    } else {
        throw new IllegalArgumentException(
        "Unknown connection type '" + connectionType + "'.");
    }   
}
/**
* Returns the sender name or null if a receiver connection.
* @return  the sender name or null if a receiver connection.
*/
public String getSenderName() {
    return senderName;
}
/**
* Returns the receiver name or null if a sender connection.
* @return  the receiver name or null if a sender connection.
*/
public String getReceiverName() {
    return receiverName;
}
/**
* Returns the sender or null if a receiver connection.
* @return  the sender or null if a receiver connection.
*/
public MessageSender getMessageSender() {
    return sender;
}
/**
* Returns the receiver or null if a sender connection.
* @return  the receiver or null if a sender connection.
*/
public MessageReceiver getMessageReceiver() {
    return receiver;
}

} // End class