package uhr.core.msg;

/**
* This class describes a single connection between parts in a
* cell and the cell's MessageRouter. It is used to connect and
* disconnect connections, as well as describe existing connections.
* It is immutable.
* <p>
* The implementation should have a no-arg constructor. This should
* not be used by clients.
*
* @author Jack Harich
*/
public interface Connection {

//---------- Public Fields ---------------------------------------
//--- Connection types for messages from parts in the router's cell
/**
* The connection type for a signal to outside of the cell.
*/
int SIGNAL_OUT   = 1;

/**
* The connection type for a signal to inside of the cell.
*/
int SIGNAL_IN    = 2;

/**
* The connection type for a service need. The service needed is
* identified by the message name.
*/
int SERVICE_NEED = 3;
//
//--- Connection types for messages to parts in the router's cell
/**
* The connection type for an interest in messages from outside of
* the cell.
*/
int INTEREST_OUT     = 4;

/**
* The connection type for an interest in messages from inside of
* the cell.
*/
int INTEREST_IN      = 5;

/**
* The connection type for a service supplier. A supplier is a part
* who can supply the service identified by the message name.
*/
int SERVICE_SUPPLIER = 6;

/**
* The connection type for a direct hookup from a sender to a
* receiver.
*/
int DIRECT           = 7;

//---------- Public Methods --------------------------------------
//----- Creators
/**
* Creates a sender connection. Senders send message signals.
*
* @param messageName     the name of the signal to connect.
* @param connectionType  the type of connection, which must be 
*                          SIGNAL_OUT, SIGNAL_IN or SERVICE_NEED.
* @param senderName      the name of the sender part.
* @param sender          the sender in the connection.
*/
public Connection createSenderConnection(
                     String        messageName,
                     int           connectionType,
                     String        senderName,
                     MessageSender sender);

/**
* Creates a receiver connection. Receivers have interests
* in particular messages.
*
* @param messageName     the name of the interest to connect.
* @param connectionType  the type of connection, which must be 
*                          INTEREST_OUT, INTEREST_IN or SERVICE_SUPPLIER.
* @param receiverName    the name of the receiver part.
* @param receiver        the receiver in the connection.
*/
public Connection createReceiverConnection(
                     String          messageName,
                     int             connectionType,
                     String          receiverName,
                     MessageReceiver receiver);

/**
* Creates a direct connection. Messages with this 
* messageName will go directly as possible from the sender to
* receiver.
*
* @param messageName     the name of the interest to connect.
* @param senderName      the name of the sender part.
* @param receiverName    the name of the receiver part.
* @param sender          the sender in the connection.
* @param receiver        the receiver in the connection.
*/
public Connection createDirectConnection(
                     String          messageName,
                     String          senderName,
                     String          receiverName,
                     MessageSender   sender,
                     MessageReceiver receiver);
                     
//----- Readers
/**
* Determines whether this is a sender connection.
* @return  true is this is a sender connection, false if not.
*/
public boolean isSenderConnection();

/**
* Determines whether this is a receiver connection.
* @return  true is this is a receiver connection, false if not.
*/
public boolean isReceiverConnection();

/**
* Determines whether this is a direct connection.
* @return  true is this is a direct connection, false if not.
*/
public boolean isDirectConnection();

/**
* Returns the message name.
* @return  the the message name.
*/
public String getMessageName();

/**
* Returns the connection type.
* @return  the connection type.
*/
public int getConnectionType();

/**
* Returns the String for the connection type constant. For example
* if the type is Connection.DIRECT then "DIRECT" is returned.
* @return  the String for the connection type constant.
*/
public String getConnectionTypeString();

/**
* Returns the connection constant given its String representation.
* For example if given "DIRECT" then Connection.DIRECT is returned.
* @return  the connection constant for the String.
*/
public int convertTypeToConstant(String connectionType);

/**
* Returns the sender name or null if a receiver connection.
* @return  the sender name or null if a receiver connection.
*/
public String getSenderName();

/**
* Returns the receiver name or null if a sender connection.
* @return  the receiver name or null if a sender connection.
*/
public String getReceiverName();

/**
* Returns the sender or null if a receiver connection.
* @return  the sender or null if a receiver connection.
*/
public MessageSender getMessageSender();

/**
* Returns the receiver or null if a sender connection.
* @return  the receiver or null if a sender connection.
*/
public MessageReceiver getMessageReceiver();


} // End interface